
#ifndef __SEGMENTATION_H
#define __SEGMENTATION_H

@begin
   include "basic.h"
   include "mem_check.h"
   include "structures.h"
   include "string.h"
   include "atomic.h"
   include "bitmap.h"
   include "image.h"
@end

/*
 * basic definition and constants
 */

// - count of extracted feature types -
const unsigned c_feature_cnt = 5;

// - extracted feature type ids -
enum {
   c_feature_id_LOCAL_MOMENTS_FEATURES = 0,
   c_feature_id_SPECTRAL_FEATURES,
   c_feature_id_COOCCURENCE_FEATURES,
   c_feature_id_FFT_FEATURES,
   c_feature_id_LBP_FEATURES,
};

// - names of extracted feature types -
extern const char *c_feature_name_strings[c_feature_cnt];

// - count of distance methods -
const unsigned c_distance_method_cnt = 5;

// - distance method ids -
enum {
   c_distance_MANHATTAN = 0,
   c_distance_UNIFORM,
   c_distance_EUCLIDEAN,
   c_distance_MINKOWSKI,
   c_distance_LOG_LIKELIHOOD,
};

// - names of distance methods -
extern const char *c_distance_method_names[c_distance_method_cnt];

/*
 * definition of structure feature_data_s
 */

extern const char *c_feature_data_version;
const unsigned c_feature_data_version_len = strlen(c_feature_data_version);

struct feature_data_s
{
   string_s img_uri;
   string_s texture_type;
   unsigned feature_id;
   string_s feature_parameters;
   unsigned width;
   unsigned height;
   unsigned fv_cnt;
   unsigned fv_length;
   bf_ptr fv_data;

   inline void init();
   inline void clear();

   inline void swap(feature_data_s &a_second);

   static inline unsigned get_feature_id_by_name(const char *a_feature_name);

   bool search_for_minimal_and_maximal(float *a_mins,float *a_maxs);
   bool normalize_by_norm_vectors(float *a_subs,float *a_divs);

   bool append_pixel_color(image_s &a_img);
   bool append_pixel_position();

   bool normalize_fvs();
   bool image_from_fvs(unsigned a_idx,image_s &a_img);
   bool save_fvs_images(const char *a_directory);

   bool create_sparse_fd(feature_data_s &a_src,unsigned a_target_fv_cnt);

   bool save_to_file(const char *a_file_name);
   bool load_from_file(const char *a_file_name);
};

/*
 * definition of structure clustering_s
 */

struct clustering_s
{
   unsigned distance_method;
   feature_data_s f_data;
   bf_arrays_s centroids;
   ui_array_s center_assign;

   inline void init();
   inline void clear();

   static float manhattan_distance(float *a_f,float *a_s,unsigned a_cnt);
   static float uniform_norm_distance(float *a_f,float *a_s,unsigned a_cnt);
   static float euclidean_distance(float *a_f,float *a_s,unsigned a_cnt);
   static float minkowski_distance(float *a_f,float *a_s,unsigned a_cnt);
   static float log_likelihood_distance(float *a_f,float *a_s,unsigned a_cnt);

   bool find_initial_centroids(unsigned a_count);
   bool k_means_distribute_vectors();
   bool k_means_adapt_centroids();
   bool k_means_run_clustering(unsigned a_max_iteration_cnt);

   bool create_segmentation_image(image_s &a_img);
};

/*
 * definition of generated structures
*/

// -- kernel_info_s --
@begin
   struct
   <
   int:x_offset
   int:y_offset
   int:x_size
   int:y_size
   >
   kernel_info_s;
@end

/*
 * definition of global functions
 */

bool erode_mask(image_s &a_img,kernel_info_s &a_ki);
bool create_index_array_from_mask(image_s &a_mask,ui_array_s &a_index_array,kernel_info_s &a_ki);
bool copy_indexed_vectors(float *a_src,float *a_trg,ui_array_s &a_index_array,unsigned a_length);
unsigned extract_masked_vectors(float *a_src,float **a_trg,image_s &a_mask,kernel_info_s &a_ki,unsigned a_length);

/*
 * inline methods of structure feature_data_s
 */

inline void feature_data_s::init()
{/*{{{*/
   img_uri.init();
   texture_type.init();
   feature_parameters.init();

   fv_data = NULL;
}/*}}}*/

inline void feature_data_s::clear()
{/*{{{*/
   img_uri.clear();
   texture_type.clear();
   feature_parameters.clear();

   if (fv_data != NULL) {
      cfree(fv_data);
      fv_data = NULL;
   }
}/*}}}*/

inline void feature_data_s::swap(feature_data_s &a_second)
{/*{{{*/
   img_uri.swap(a_second.img_uri);
   texture_type.swap(a_second.texture_type);

   unsigned tmp_unsigned = feature_id;
   feature_id = a_second.feature_id;
   a_second.feature_id = tmp_unsigned;

   feature_parameters.swap(a_second.feature_parameters);

   tmp_unsigned = width;
   width = a_second.width;
   a_second.width = tmp_unsigned;

   tmp_unsigned = height;
   height = a_second.height;
   a_second.height = tmp_unsigned;

   tmp_unsigned = fv_cnt;
   fv_cnt = a_second.fv_cnt;
   a_second.fv_cnt = tmp_unsigned;

   tmp_unsigned = fv_length;
   fv_length = a_second.fv_length;
   a_second.fv_length = tmp_unsigned;

   float *tmp_float_ptr = fv_data;
   fv_data = a_second.fv_data;
   a_second.fv_data = tmp_float_ptr;
}/*}}}*/

inline unsigned feature_data_s::get_feature_id_by_name(const char *a_feature_name)
{/*{{{*/
   unsigned idx = 0;
   do {
      if (strcmp(c_feature_name_strings[idx],a_feature_name) == 0) {
         return idx;
      }
   } while(++idx < c_feature_cnt);

   return c_idx_not_exist;
}/*}}}*/

/*
 * inline methods of structure clustering_s
 */

inline void clustering_s::init()
{
   distance_method = c_distance_MANHATTAN;
   f_data.init();
   centroids.init();
   center_assign.init();
}

inline void clustering_s::clear()
{
   f_data.clear();
   centroids.clear();
   center_assign.clear();
}

/*
 * inline methods of generated structures
 */

// -- kernel_info_s --
@begin
   inlines kernel_info_s
@end

#endif

