
@begin
   include "segmentation.h"
@end

/*
 * basic definitions and constants
 */

// - names of extracted feature types -
const char *c_feature_name_strings[c_feature_cnt] = {
   "LOCAL_MOMENTS_0000",
   "SPECTRAL_0000",
   "COOCCURENCE_0000",
   "FFT_GABOR_0000",
   "LBP_0000",
};

// - names of distance methods -
const char *c_distance_method_names[c_distance_method_cnt] = {
   "MANHATTAN",
   "UNIFORM",
   "EUCLIDEAN",
   "MINKOWSKI",
   "LOG_LIKELIHOOD",
};

/*
 * methods of structure feature_data_s
 */

const char *c_feature_data_version = "FVD_0000";

bool feature_data_s::search_for_minimal_and_maximal(float *a_mins,float *a_maxs)
{/*{{{*/
   if (fv_cnt == 0 || fv_length == 0 || fv_data == NULL) {
      return false;
   }

   // - find minimal and maximal values -
   {
      unsigned f_idx = 0;
      do {
         float &min_value = a_mins[f_idx];
         float &max_value = a_maxs[f_idx];

         // - cycle throught all hist data values -
         float *d_ptr = fv_data + f_idx;
         float *d_ptr_end = d_ptr + fv_cnt*fv_length;
         do {
            *d_ptr < min_value?min_value = *d_ptr:0;
            *d_ptr > max_value?max_value = *d_ptr:0;
         } while((d_ptr += fv_length) < d_ptr_end);
      } while(++f_idx < fv_length);
   }

   return true;
}/*}}}*/

bool feature_data_s::normalize_by_norm_vectors(float *a_subs,float *a_divs)
{/*{{{*/
   if (fv_cnt == 0 || fv_length == 0 || fv_data == NULL) {
      return false;
   }

   // - normalize values -
   {
      unsigned f_idx = 0;
      do {
         float sub_value = a_subs[f_idx];
         float div_value = a_divs[f_idx];

         // - cycle throught all hist data values -
         float *d_ptr = fv_data + f_idx;
         float *d_ptr_end = d_ptr + fv_cnt*fv_length;
         do {
            *d_ptr = (*d_ptr - sub_value)/div_value;
         } while((d_ptr += fv_length) < d_ptr_end);
      } while(++f_idx < fv_length);
   }

   return true;
}/*}}}*/

bool feature_data_s::append_pixel_color(image_s &a_img)
{/*{{{*/
   if (width > a_img.width || height > a_img.height || fv_cnt != width*height || a_img.pixel_format != c_image_pixel_format_3x8U) {
      return false;
   }

   unsigned n_fv_length = fv_length + 3;
   float *n_fv_data = (float *)cmalloc(fv_cnt*n_fv_length*sizeof(float));

   // - create referred image -
   image_s r_img;
   r_img.init();
   r_img.create_referred((a_img.width - width) >> 1,(a_img.height - height) >> 1,width,height,a_img);
   
   unsigned r_pixel_step = r_img.pixel_step;
   unsigned r_image_ls = r_img.width*r_pixel_step;
   unsigned r_line_size = r_img.image_data_ptr->line_bytes;

   // - add color information to feature vectors -
   unsigned char *r_ptr = r_img.image_data_ptr->data + r_img.y_pos*r_line_size + r_img.x_pos*r_pixel_step;
   unsigned char *r_ptr_end = r_ptr + (r_img.height - 1)*r_line_size + r_img.width*r_pixel_step;
   float *ptr = fv_data;
   float *n_ptr = n_fv_data;
   do {
      unsigned char *r_ptr_w_end = r_ptr + r_image_ls;
      do {
         memcpy(n_ptr,ptr,fv_length*sizeof(float));

         // - set pixel color -
         n_ptr[fv_length] = r_ptr[0]/255.0f;
         n_ptr[fv_length + 1] = r_ptr[1]/255.0f;
         n_ptr[fv_length + 2] = r_ptr[2]/255.0f;
      } while(n_ptr += n_fv_length,ptr += fv_length,(r_ptr += r_pixel_step) < r_ptr_w_end);

      r_ptr += r_line_size - r_image_ls;
   } while(r_ptr < r_ptr_end);

   r_img.clear();

   cfree(fv_data);

   fv_length = n_fv_length;
   fv_data = n_fv_data;

   // - insert information to feature_parameters -
   feature_parameters.concf(" pixel_color ");

   return true;
}/*}}}*/

bool feature_data_s::append_pixel_position()
{/*{{{*/
   if (width == 0 || height == 0 || fv_cnt != width*height) {
      return false;
   }

   // - allocate memory for new fv_data -
   unsigned n_fv_length = fv_length + 2;
   float *n_fv_data = (float *)cmalloc(fv_cnt*n_fv_length*sizeof(float));

   // - increment values -
   float x_inc_value = 1.0f/width;
   float y_inc_value = 1.0f/height;

   float *ptr = fv_data;
   float *ptr_end = ptr + fv_cnt*fv_length;
   float *n_ptr = n_fv_data;
   float y_value = 0.0f;
   do {
      float *ptr_w_end = ptr + width*fv_length;
      float x_value = 0.0f;
      do {
         memcpy(n_ptr,ptr,fv_length*sizeof(float));

         // - set pixel position -
         n_ptr[fv_length] = x_value;
         n_ptr[fv_length + 1] = y_value;
      } while(x_value += x_inc_value,n_ptr += n_fv_length,(ptr += fv_length) < ptr_w_end);
   } while(y_value += y_inc_value,ptr < ptr_end);

   cfree(fv_data);

   fv_length = n_fv_length;
   fv_data = n_fv_data;

   // - insert information to feature_parameters -
   feature_parameters.concf(" pixel_position ");

   return true;
}/*}}}*/

bool feature_data_s::normalize_fvs()
{/*{{{*/
   if (fv_cnt == 0 || fv_length == 0 || fv_data == NULL) {
      return false;
   }

   float min_values[fv_length];
   float max_values[fv_length];

   // - initialize minimal and maximal values -
   {
      float *min_ptr = min_values;
      float *max_ptr = max_values;
      float *min_ptr_end = min_ptr + fv_length;
      do {
         *min_ptr = INFINITY;
         *max_ptr = -INFINITY;
      } while(++max_ptr,++min_ptr < min_ptr_end);
   }

   // - find minimal and maximal values -
   {
      unsigned f_idx = 0;
      do {
         float &min_value = min_values[f_idx];
         float &max_value = max_values[f_idx];

         // - cycle throught all hist data values -
         float *d_ptr = fv_data + f_idx;
         float *d_ptr_end = d_ptr + fv_cnt*fv_length;
         do {
            *d_ptr < min_value?min_value = *d_ptr:0;
            *d_ptr > max_value?max_value = *d_ptr:0;
         } while((d_ptr += fv_length) < d_ptr_end);
      } while(++f_idx < fv_length);
   }

   // - normalize values -
   {
      unsigned f_idx = 0;
      do {
         float &min_value = min_values[f_idx];
         float &max_value = max_values[f_idx];

         if (max_value > min_value) {

            float diff_value = max_value - min_value;

            // - cycle throught all hist data values -
            float *d_ptr = fv_data + f_idx;
            float *d_ptr_end = d_ptr + fv_cnt*fv_length;
            do {
               *d_ptr = (*d_ptr - min_value)/diff_value;
            } while((d_ptr += fv_length) < d_ptr_end);
         }
      } while(++f_idx < fv_length);
   }

   return true;
}/*}}}*/

bool feature_data_s::image_from_fvs(unsigned a_idx,image_s &a_img)
{/*{{{*/
   if (a_img.pixel_format != c_image_pixel_format_32F || a_img.width != width
   || a_img.height != height || fv_data == NULL || fv_length == 0 || a_idx >=
   fv_length) {
      return false;
   }

   unsigned pixel_step = a_img.pixel_step;
   unsigned line_size = a_img.image_data_ptr->line_bytes;
   unsigned image_ls = a_img.width*pixel_step;

   unsigned char *ptr = a_img.image_data_ptr->data + a_img.y_pos*line_size + a_img.x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (a_img.height - 1)*line_size + image_ls;
   float *v_ptr = fv_data + a_idx;
   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
         *((float *)ptr) = *v_ptr;
      } while(v_ptr += fv_length,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

bool feature_data_s::save_fvs_images(const char *a_directory)
{/*{{{*/
   if (a_directory == NULL || fv_data == NULL || width == 0 || height == 0) {
      return false;
   }

   image_s img_32f;
   img_32f.init();
   img_32f.create(width,height,c_image_pixel_format_32F);

   image_s img_3x8u;
   img_3x8u.init();
   img_3x8u.create(width,height,c_image_pixel_format_3x8U);

   string_s file_name;
   file_name.init();
   
   unsigned f_pos_idx = 0;
   do {
      cassert(image_from_fvs(f_pos_idx,img_32f));
      cassert(img_3x8u.io_convert(img_32f));

      file_name.setf("%s/img_%4.4d.bmp",a_directory,f_pos_idx);
      img_3x8u.save_to_bmp_file(file_name.data);
   } while(++f_pos_idx < fv_length);

   file_name.clear();
   img_3x8u.clear();
   img_32f.clear();

   return true;
}/*}}}*/

bool feature_data_s::create_sparse_fd(feature_data_s &a_src,unsigned a_target_fv_cnt)
{/*{{{*/
   if (a_src.fv_data == NULL || a_src.width != 0 || a_src.height != 0 || a_src.fv_cnt == 0 || a_target_fv_cnt > a_src.fv_cnt) {
      return false;
   }

   // - clear feature data -
   clear();

   // - copy image, texture and feature informations -
   img_uri = a_src.img_uri;
   texture_type = a_src.texture_type;
   feature_id = a_src.feature_id;
   feature_parameters = a_src.feature_parameters;
   width = height = 0;
   fv_cnt = a_target_fv_cnt;
   fv_length = a_src.fv_length;

   fv_data = (float *)cmalloc(fv_cnt*fv_length*sizeof(float));

   unsigned accumulator = 0;

   float *fv_ptr = fv_data;
   float *fv_ptr_end = fv_ptr + fv_cnt*fv_length;
   float *sfv_ptr = a_src.fv_data;
   do {
      memcpy(fv_ptr,sfv_ptr,fv_length*sizeof(float));

      accumulator += a_src.fv_cnt;
      sfv_ptr += (accumulator / fv_cnt)*a_src.fv_length;
      accumulator %= fv_cnt;
   } while((fv_ptr += fv_length) < fv_ptr_end);

   return true;
}/*}}}*/

bool feature_data_s::save_to_file(const char *a_file_name)
{/*{{{*/
   if (fv_data == NULL || feature_id >= c_feature_cnt || a_file_name == NULL) {
      return false;
   }

   if (width == 0 || height == 0) {
      if (fv_cnt == 0) {
         return false;
      }

      width = height = 0;
   }
   else {
      cassert(width*height == fv_cnt);
   }

   FILE *f = fopen(a_file_name,"wb");
   if (f == NULL) return false;

   // - write file identifie -
   fwrite(c_feature_data_version,sizeof(char),c_feature_data_version_len,f);
   
   // - write source image url -
   unsigned img_uri_len = img_uri.size - 1;
   fwrite(&img_uri_len,sizeof(unsigned),1,f);
   fwrite(img_uri.data,sizeof(char),img_uri_len,f);

   // - write texture type info -
   unsigned texture_type_len = texture_type.size - 1;
   fwrite(&texture_type_len,sizeof(unsigned),1,f);
   fwrite(texture_type.data,sizeof(char),texture_type_len,f);

   // - write feature identification -
   unsigned feature_id_len = strlen(c_feature_name_strings[feature_id]);
   fwrite(&feature_id_len,sizeof(unsigned),1,f);
   fwrite(c_feature_name_strings[feature_id],sizeof(char),feature_id_len,f);

   // - write feature parameters -
   unsigned fp_len = feature_parameters.size - 1;
   fwrite(&fp_len,sizeof(unsigned),1,f);
   fwrite(feature_parameters.data,sizeof(char),fp_len,f);

   // - write feature vectors data -
   fwrite(&width,sizeof(unsigned),1,f);
   fwrite(&height,sizeof(unsigned),1,f);
   fwrite(&fv_cnt,sizeof(unsigned),1,f);
   fwrite(&fv_length,sizeof(unsigned),1,f);
   fwrite(fv_data,sizeof(float),fv_cnt*fv_length,f);

   fclose(f);

   return true;
}/*}}}*/

bool feature_data_s::load_from_file(const char *a_file_name)
{/*{{{*/
   if (a_file_name == NULL) {
      return false;
   }

   clear();

   FILE *f = fopen(a_file_name,"rb");
   if (f == NULL) {
      return false;
   }

   char feature_data_version[c_feature_data_version_len + 1];

   cassert(fread(feature_data_version,sizeof(char),c_feature_data_version_len,f) == c_feature_data_version_len);
   feature_data_version[c_feature_data_version_len] = '\0';

   if (strcmp(c_feature_data_version,feature_data_version) != 0) {
      fclose(f);
      return false;
   }

   unsigned img_uri_len;
   cassert(fread(&img_uri_len,sizeof(unsigned),1,f) == 1);
   img_uri.create(img_uri_len);
   cassert(fread(img_uri.data,sizeof(char),img_uri_len,f) == img_uri_len);

   unsigned texture_type_len;
   cassert(fread(&texture_type_len,sizeof(unsigned),1,f) == 1);
   texture_type.create(texture_type_len);
   cassert(fread(texture_type.data,sizeof(char),texture_type_len,f) == texture_type_len);

   unsigned feature_id_len;
   cassert(fread(&feature_id_len,sizeof(unsigned),1,f) == 1);
   char feature_id_str[feature_id_len + 1];
   feature_id_str[feature_id_len] = '\0';
   cassert(fread(feature_id_str,sizeof(char),feature_id_len,f) == feature_id_len);

   feature_id = get_feature_id_by_name(feature_id_str);
   if (feature_id == c_idx_not_exist) {
      clear();
      fclose(f);
      return false;
   }

   // - write feature parameters -
   unsigned fp_len;
   cassert(fread(&fp_len,sizeof(unsigned),1,f) == 1);
   feature_parameters.create(fp_len);
   cassert(fread(feature_parameters.data,sizeof(char),fp_len,f) == fp_len);

   cassert(fread(&width,sizeof(unsigned),1,f) == 1);
   cassert(fread(&height,sizeof(unsigned),1,f) == 1);
   cassert(fread(&fv_cnt,sizeof(unsigned),1,f) == 1);
   cassert(fread(&fv_length,sizeof(unsigned),1,f) == 1);

   fv_data = (float *)cmalloc(fv_cnt*fv_length*sizeof(float));
   cassert(fread(fv_data,sizeof(float),fv_cnt*fv_length,f) == fv_cnt*fv_length);

   fclose(f);

   return true;
}/*}}}*/

/*
 * methods of structure clustering_s
 */

float clustering_s::manhattan_distance(float *a_f,float *a_s,unsigned a_length)
{/*{{{*/
   float distance = 0.0f;

   float *f_ptr = a_f;
   float *f_ptr_end = f_ptr + a_length;
   float *s_ptr = a_s;
   do {
      register float value = *f_ptr - *s_ptr;
      distance += value < 0?-value:value;
   } while(++s_ptr,++f_ptr < f_ptr_end);

   return distance;
}/*}}}*/

float clustering_s::uniform_norm_distance(float *a_f,float *a_s,unsigned a_length)
{/*{{{*/
   float max = 0.0f;

   float *f_ptr = a_f;
   float *f_ptr_end = f_ptr + a_length;
   float *s_ptr = a_s;
   do {
      register float value = *f_ptr - *s_ptr;
      value = value < 0.0f?-value:value;
      max < value?max=value:0;
   } while(++s_ptr,++f_ptr < f_ptr_end);

   return max;
}/*}}}*/

float clustering_s::euclidean_distance(float *a_f,float *a_s,unsigned a_length)
{/*{{{*/
   float sum = 0.0f;

   float *f_ptr = a_f;
   float *f_ptr_end = f_ptr + a_length;
   float *s_ptr = a_s;
   do {
      register float value = *f_ptr - *s_ptr;
      sum += value*value;
   } while(++s_ptr,++f_ptr < f_ptr_end);

   return sqrtf(sum);
}/*}}}*/

float clustering_s::minkowski_distance(float *a_f,float *a_s,unsigned a_length)
{/*{{{*/
   float sum = 0.0f;

   float *f_ptr = a_f;
   float *f_ptr_end = f_ptr + a_length;
   float *s_ptr = a_s;
   do {
      register float value = *f_ptr - *s_ptr;
      value = value < 0.0f?-value:value;

      sum += powf(value,(float)a_length);
   } while(++s_ptr,++f_ptr < f_ptr_end);

   return powf(sum,1/(float)a_length);
}/*}}}*/

float clustering_s::log_likelihood_distance(float *a_f,float *a_s,unsigned a_length)
{/*{{{*/
   const float c_2log2 = 2.0f*logf(2.0f);
   float result = c_2log2;

   float *f_ptr = a_f;
   float *f_ptr_end = f_ptr + a_length;
   float *s_ptr = a_s;

   do {
      result += *f_ptr*logf(*f_ptr) + *s_ptr*logf(*s_ptr) - (*f_ptr + *s_ptr)*logf(*f_ptr + *s_ptr);
   } while(++s_ptr,++f_ptr < f_ptr_end);

   return result;
}/*}}}*/

bool clustering_s::find_initial_centroids(unsigned a_count)
{/*{{{*/
   if (f_data.fv_data == NULL || a_count == 0) {
      return false;
   }

   centroids.used = 0;
   centroids.copy_resize(a_count);

   // - create random centroids -
   do {
      centroids.push_blank();
      bf_array_s &centroid = centroids.last();

      centroid.used = 0;
      centroid.copy_resize(f_data.fv_length);
      centroid.used = f_data.fv_length;

      float *f_ptr = centroid.data;
      float *f_ptr_end = f_ptr + centroid.used;
      do {
         *f_ptr = (float)rand()/(float)RAND_MAX;
      } while(++f_ptr < f_ptr_end);
   } while(centroids.used < a_count);

   return true;
}/*}}}*/

bool clustering_s::k_means_distribute_vectors()
{/*{{{*/
   cassert(centroids.used != 0 && f_data.fv_data != NULL);

   // - true if segmentation changed -
   bool change = false;

   // - create center align vector -
   if (center_assign.used != f_data.fv_cnt) {
      center_assign.used = 0;
      center_assign.copy_resize(f_data.fv_cnt);
      center_assign.used = f_data.fv_cnt;

      // - initialize center align values -
      memset(center_assign.data,0,center_assign.used*sizeof(float));
   }

   float *vd_ptr = f_data.fv_data;
   float *vd_ptr_end = vd_ptr + f_data.fv_cnt*f_data.fv_length;
   unsigned *ca_ptr = center_assign.data;

   // - compute distances -
   do {
      float min_distance = INFINITY;
      unsigned min_distance_idx = 0;

      unsigned c_idx = 0;
      do {
         bf_array_s &centroid = centroids[c_idx];

         register float value;

         switch (distance_method) {
         case c_distance_MANHATTAN:
            value = manhattan_distance(vd_ptr,centroid.data,f_data.fv_length);
            break;
         case c_distance_UNIFORM:
            value = uniform_norm_distance(vd_ptr,centroid.data,f_data.fv_length);
            break;
         case c_distance_EUCLIDEAN:
            value = euclidean_distance(vd_ptr,centroid.data,f_data.fv_length);
            break;
         case c_distance_MINKOWSKI:
            value = minkowski_distance(vd_ptr,centroid.data,f_data.fv_length);
            break;
         case c_distance_LOG_LIKELIHOOD:
            value = log_likelihood_distance(vd_ptr,centroid.data,f_data.fv_length);
            break;
         default:
            cassert(0);
         }

         if (min_distance > value) {
            min_distance = value;
            min_distance_idx = c_idx;
         }
      } while(++c_idx < centroids.used);

      // - set center align to centroid with minimal distance -
      if (*ca_ptr != min_distance_idx) {
         *ca_ptr = min_distance_idx;
         change = true;
      }
   } while(++ca_ptr,(vd_ptr += f_data.fv_length) < vd_ptr_end);

   return change;
}/*}}}*/

bool clustering_s::k_means_adapt_centroids()
{/*{{{*/
   if (f_data.fv_data == NULL || centroids.used == 0 || center_assign.used == 0) {
      return false;
   }

   // - clear centroids values -
   {
      bf_array_s *c_ptr = centroids.data;
      bf_array_s *c_ptr_end = c_ptr + centroids.used;
      do {
         memset(c_ptr->data,0,f_data.fv_length*sizeof(float));
      } while(++c_ptr < c_ptr_end);
   }

   unsigned contrib_cnt[centroids.used];
   memset(contrib_cnt,0,centroids.used*sizeof(unsigned));

   // - compute mean values of centroids -
   float *vd_ptr = f_data.fv_data;
   float *vd_ptr_end = vd_ptr + f_data.fv_cnt*f_data.fv_length;
   unsigned *ca_ptr = center_assign.data;
   do {
      bf_array_s &centroid = centroids[*ca_ptr];
      contrib_cnt[*ca_ptr]++;

      // - add vector values to centroid -
      float *c_ptr = centroid.data;
      float *c_ptr_end = c_ptr + f_data.fv_length;
      do {
         *c_ptr += *vd_ptr;
      } while(++vd_ptr,++c_ptr < c_ptr_end);
   } while(++ca_ptr,vd_ptr < vd_ptr_end);

   // - divide centroid values -
   {
      unsigned c_idx = 0;
      do {
         if (contrib_cnt[c_idx] != 0) {
            float div_value = (float)contrib_cnt[c_idx];
            
            float *f_ptr = centroids[c_idx].data;
            float *f_ptr_end = f_ptr + f_data.fv_length;
            do {
               *f_ptr /= div_value;
            } while(++f_ptr < f_ptr_end);
         }
         else {
            float *f_ptr = centroids[c_idx].data;
            float *f_ptr_end = f_ptr + f_data.fv_length;
            do {
               *f_ptr = (float)rand()/(float)RAND_MAX;
            } while(++f_ptr < f_ptr_end);
         }
      } while(++c_idx < centroids.used);
   }

   return true;
}/*}}}*/

bool clustering_s::k_means_run_clustering(unsigned a_max_iteration_cnt)
{/*{{{*/
   unsigned idx = 0;
   do {
      if (!k_means_distribute_vectors()) break;
      if (!k_means_adapt_centroids()) {
         return false;
      }

      // FIXME progress output
      fprintf(stderr,"%u ",idx);
   } while(++idx < a_max_iteration_cnt);

   // FIXME progress output
   fputc('\n',stderr);

   return true;
}/*}}}*/

bool clustering_s::create_segmentation_image(image_s &a_img)
{/*{{{*/
   if (center_assign.used == 0 || f_data.width == 0 || f_data.height == 0) {
      return false;
   }

   a_img.clear();
   a_img.create(f_data.width,f_data.height,c_image_pixel_format_32U);

   unsigned pixel_step = a_img.pixel_step;
   unsigned line_size = a_img.image_data_ptr->line_bytes;
   unsigned image_ls = a_img.width*pixel_step;

   unsigned char *ptr = a_img.image_data_ptr->data + a_img.y_pos*line_size + a_img.x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (a_img.height - 1)*line_size + image_ls;
   unsigned *ca_ptr = center_assign.data;

   // - set image values by center_assign distribution -
   do {
      unsigned char *ptr_w_end = ptr + image_ls;
      do {
         *((unsigned *)ptr) = *ca_ptr;
      } while(++ca_ptr,(ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
   } while(ptr < ptr_end);

   return true;
}/*}}}*/

/*
 * methods of generated structures
 */

// -- kernel_info_s --
@begin
   methods kernel_info_s
@end

/*
 * global functions
 */

bool erode_mask(image_s &a_img,kernel_info_s &a_ki)
{/*{{{*/
   if (a_img.pixel_format != c_image_pixel_format_8U) {
      return false;
   }

   unsigned kx_size = (a_ki.x_size << 1) + 1;
   unsigned ky_size = (a_ki.y_size << 1) + 1;
   unsigned kx_center = kx_size >> 1;
   unsigned ky_center = ky_size >> 1;

   image_s kernel;
   kernel.init();
   kernel.create(kx_size,ky_size,c_image_pixel_format_8U);

   unsigned char black_color = 0;
   kernel.io_fill(&black_color);

   // - fill in mask -
   {
      image_s r_kernel;
      r_kernel.init();
      r_kernel.create_referred(kx_center + a_ki.x_offset,ky_center + a_ki.y_offset,a_ki.x_size,a_ki.y_size,kernel);

      unsigned char white_color = 255;
      r_kernel.io_fill(&white_color);

      r_kernel.clear();
   }

   image_s tmp_img;
   tmp_img.init();
   tmp_img.create(a_img);

   if (!tmp_img.io_morphology(a_img,kernel,c_image_operator_binary_erode)) {
      return false;
   }

   a_img.swap(tmp_img);

   tmp_img.clear();
   kernel.clear();

   return true;
}/*}}}*/

bool create_index_array_from_mask(image_s &a_mask,ui_array_s &a_index_array,kernel_info_s &a_ki)
{/*{{{*/
   if (a_mask.pixel_format != c_image_pixel_format_8U) {
      return false;
   }

   image_s r_mask;
   r_mask.init();
   r_mask.create_referred(-a_ki.x_offset,-a_ki.y_offset,a_mask.width - (a_ki.x_size - 1),a_mask.height - (a_ki.y_size - 1),a_mask);

   a_index_array.used = 0;

   unsigned pixel_step = r_mask.pixel_step;
   unsigned image_ls = r_mask.width*pixel_step;
   unsigned line_size = r_mask.image_data_ptr->line_bytes;

   unsigned char *ptr = r_mask.image_data_ptr->data + r_mask.y_pos*line_size + r_mask.x_pos*pixel_step;
   unsigned char *ptr_end = ptr + (r_mask.height - 1)*line_size + r_mask.width*pixel_step;
   unsigned fv_idx = 0;
   do {
      unsigned char *ptr_w_end = ptr + image_ls;

      do {
         if (*ptr != 0) {
            a_index_array.push(fv_idx);
         }

         fv_idx++;
      } while((ptr += pixel_step) < ptr_w_end);

      ptr += line_size - image_ls;
   } while(ptr < ptr_end);

   r_mask.clear();

   return true;
}/*}}}*/

bool copy_indexed_vectors(float *a_src,float *a_trg,ui_array_s &a_index_array,unsigned a_length)
{/*{{{*/
   if (a_index_array.used > 0) {
      unsigned *i_ptr = a_index_array.data;
      unsigned *i_ptr_end = i_ptr + a_index_array.used;

      float *t_ptr = a_trg;
      do {
         memcpy(t_ptr,a_src + *i_ptr*a_length,a_length*sizeof(float));
      } while(t_ptr += a_length,++i_ptr < i_ptr_end);
   }

   return true;
}/*}}}*/

unsigned extract_masked_vectors(float *a_src,float **a_trg,image_s &a_mask,kernel_info_s &a_ki,unsigned a_length)
{/*{{{*/

   if (a_ki.x_size > 1 || a_ki.y_size > 1) {
      if (!erode_mask(a_mask,a_ki)) {
         return c_idx_not_exist;
      }
   }

   ui_array_s index_array;
   index_array.init();

   if (!create_index_array_from_mask(a_mask,index_array,a_ki)) {
      index_array.clear();
      return c_idx_not_exist;
   }

   *a_trg = (float *)cmalloc(index_array.used*a_length*sizeof(float));

   if (!copy_indexed_vectors(a_src,*a_trg,index_array,a_length)) {
      index_array.clear();
      cfree(*a_trg);
      *a_trg = NULL;

      return c_idx_not_exist;
   }

   unsigned index_array_length = index_array.used;
   index_array.clear();

   return index_array_length;
}/*}}}*/

