
#ifndef __FFT_SEG_H
#define __FFT_SEG_H

@begin
   include "basic.h"
   include "mem_check.h"
   include "structures.h"
   include "atomic.h"
   include "bitmap.h"
   include "image.h"
   include "segmentation.h"
@end

/*
 * basic definition and constants
 */

const unsigned c_gf_freq_cnt = 1;
const unsigned c_gf_rotation_cnt = 8;

/*
 * definition of generated structures
 */

// -- fft_seg_s --
@begin
   struct
   <
   unsigned:freq_cnt
   unsigned:rotation_cnt
   unsigned:filter_bank_size
   unsigned:gabor_response_size

   unsigned:spectrum_w_power
   unsigned:spectrum_h_power
   image_array_s:filter_bank
   image_s:spectrum
   image_array_s:responses
   image_array_s:nms_responses

   unsigned:width
   unsigned:height
   unsigned:length
   bf_ptr:seg_data
   >

   not_generate: init clear

   additions {
      inline void init();
      inline void clear();
      inline void setup(unsigned a_freq_cnt,unsigned a_rotation_cnt);

      bool create_filter_bank(unsigned a_w_power,unsigned a_h_power);
      inline bool compute_spectrum(image_s &a_img);
      bool compute_responses_to_filter_bank();

      bool responses_to_32F();
      bool non_maxima_suppression(unsigned a_block_size);
      bool directional_non_maxima_suppression();
      bool reconstruction_test();
      bool get_nms_responses_max(image_s &a_img);

      bool compute_seg_data_from_responses(unsigned a_width,unsigned a_height);

      float compute_norm_matrix_distance(float *a_f,float *a_s);
      bool create_image_from_data_distances(image_s &a_img);

      void get_feature_parameters_string(string_s &a_fp_str);
      inline bool get_features(feature_data_s &a_fd);
      bool get_features_MASK(image_s &a_mask,feature_data_s &a_fd);
   }

   fft_seg_s;
@end

/*
 * inline methods of structure image_s
 */

// -- fft_seg_s --
@begin
   inlines fft_seg_s
@end

inline void fft_seg_s::init()
{/*{{{*/
   filter_bank.init();
   spectrum.init();
   responses.init();
   nms_responses.init();
   seg_data = NULL;
}/*}}}*/

inline void fft_seg_s::clear()
{/*{{{*/
   filter_bank.clear();
   spectrum.clear();
   responses.clear();
   nms_responses.clear();

   if (seg_data != NULL) {
      cfree(seg_data);
      seg_data = NULL;
   }

   init();
}/*}}}*/

inline void fft_seg_s::setup(unsigned a_freq_cnt,unsigned a_rotation_cnt)
{
   freq_cnt = a_freq_cnt;
   rotation_cnt = a_rotation_cnt;
   filter_bank_size = freq_cnt*rotation_cnt;
   gabor_response_size = filter_bank_size;

}

inline bool fft_seg_s::compute_spectrum(image_s &a_img)
{/*{{{*/
   if (a_img.pixel_format == c_image_pixel_format_blank || a_img.width != 1U << spectrum_w_power || 
      a_img.height != 1U << spectrum_h_power) {
      return false;
   }

   // - create spectrum image -
   if (spectrum.pixel_format == c_image_pixel_format_blank || spectrum.width != a_img.width || spectrum.height != a_img.height) {
      spectrum.create(a_img.width,a_img.height,c_image_pixel_format_2x64F);
   }
   
   // - convert image to spectrum format -
   if (!spectrum.io_convert(a_img)) {
      return false;
   }

   // - compute 2D fft -
   cassert(spectrum.io_compute_fft(true,spectrum_w_power,spectrum_h_power));
   cassert(spectrum.io_swap_quarters());

   return true;
}/*}}}*/

inline bool fft_seg_s::get_features(feature_data_s &a_fd)
{/*{{{*/
   a_fd.feature_id = c_feature_id_FFT_FEATURES;

   get_feature_parameters_string(a_fd.feature_parameters);

   a_fd.width = width;
   a_fd.height = height;
   a_fd.fv_cnt = width*height;
   a_fd.fv_length = length;
   a_fd.fv_data = seg_data;
   seg_data = NULL;

   return true;
}/*}}}*/

#endif

