#!/usr/bin/python
#coding=utf8

from twisted.web2 import server, http, resource, channel, static, http_headers, log
from twisted.web2.stream import readStream
from twisted.application import service, internet, strports

import time
import re

from __base  import *
from __pgsql import *
from __process_query import *

exp_time_re = re.compile(r'\d{4}-\d{2}-\d{2},\d{2}:\d{2}:\d{2}')
exp_loc_re = re.compile(r'[N|S]\d{2}\.\d{2}:\d{2}\.\d{2}:\d{2}\.\d{2},[E|W]\d{3}\.\d{2}:\d{2}\.\d{2}:\d{2}\.\d{2}')
radius_re = re.compile(r'(\d+(\.\d+)?)')

## =============================================================================
# File handling
#
class File:

   ## ---------------------------------------------------------------------------
   # Writes 'data' into disk file called 'name'.
   # @param path Path to file.
   # @param data Data to be written into file.
   def save(self,path,data):
      f = open(path,'w')
      f.write(data)
      f.close
	
	
   ## ---------------------------------------------------------------------------
   # Writes 'stream' into disk file called 'name' using internal callback function
   # and calling another callback function after 'stream' has been saved.
   # @param path Path to file.
   # @param stream Data stream to be written into file.
   # @param after_save_function Function to be called after stream has been saved.
   def save_stream(self,path,stream,after_save_function):
      f = open(path,'w')

      # Callback function for saving the stream
      def gotData(d):
         f.write(d)

      # Callback function to be called after the stream has been saved
      def _finishedReading(ignore):
         f.close()
         after_save_function() # call function which has to be called after file has been saved

      d = readStream(stream,gotData)			
      d.addCallback(_finishedReading)
	
   ## ---------------------------------------------------------------------------
   # Opens 'path' disk file and returns file descriptor to it.
   # @param path Path to file.
   # @return File descriptor or None in case of failure.
   def load_stream(self,path):
      try:
         f = open(path,'r')
      except IOError:
         return None
      except:
         return None
            
      return f

## =============================================================================
#
#
class ExampleResource(resource.Resource):
   def allowedMethods(self):
      return ('GET',)

   def http_GET(self,request):
      arg_num = request.args.get("num",[])

      ret_string = "Test return page, num = \"%s\"\n" % arg_num
      return http.Response(202,{'Content-Type':http_headers.MimeType('text','plain')},stream=ret_string)

## =============================================================================
# Class handling image files.
#
class ImageFileResource(resource.Resource):

   def allowedMethods(self):
      return ('PUT','GET','DELETE')

   ## ---------------------------------------------------------------------------
   # Constructor
   # @param f
   def __init__ (self,f):
      self.file = f

   ## ---------------------------------------------------------------------------
   # Upload image (generate unique URI).
   # @param request 
   def http_PUT(self,request):
      log_msg("ImageFileResource::http_PUT")
	
      # - check content type -
      contentType = request.headers.getHeader('Content-Type')
      contentType = str(contentType)
      if not contentType.startswith("MimeType('image'"):
         ret_string = "Wrong Content-Type, accepts only 'image' MIME type.\n"
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # get parameter exp_time - expected format: yyyy-mm-dd,hh:mm:ss
      exp_time = request.args.get("exp_time",None)
      if exp_time == None:
         s_exp_time = "NULL"
      else:
         try:
            time.strptime(exp_time[0],'%Y-%m-%d,%H:%M:%S') 
         except:
            ret_string = "Wrong format of parameter 'exp_time', expected format 'yyyy-mm-dd,hh:mm:ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

         s_exp_time = "'%s'" % exp_time[0].replace(","," ")

      # get parameter exp_loc - expected format: [N|S]dd.dd:mm.mm:ss.ss,[E|W]ddd.dd:mm.mm:ss.ss
      exp_loc = request.args.get("exp_loc",None)
      if exp_loc == None:
         s_exp_loc = "NULL"
      else:
         re_subs = exp_loc_re.findall(exp_loc[0])
         if len(re_subs) == 1 and re_subs[0] == exp_loc[0]:
            s_exp_loc = DBin_geo_point(exp_loc[0])
            s_exp_loc = "ST_GeographyFromText('SRID=4326; %s')" % s_exp_loc
         else:
            ret_string = "Wrong format of parameter 'exp_loc', expected format '[N|S]dd.dd:mm.mm:ss.ss,[E|W]ddd.dd:mm.mm:ss.ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - extract image type -
      image_type = contentType[len("MimeType('image', '"):]
      image_type = image_type[:image_type.find("'")]

      # - 1) insert image with void uri and state set to "inserted" in DB, not uploaded yet -
      f_pgsql_execute("INSERT INTO images (id,uri,exp_time,exp_loc,state,video_id) VALUES (DEFAULT,'',%s,%s,%d,0)" % (s_exp_time,s_exp_loc,IMAGE_state_inserted))
      if db_cursor.rowcount != 1:
         return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())
		
      # - 2) retrieve image id -
      f_pgsql_execute("SELECT MAX(id) FROM images")
      if db_cursor.rowcount != 1:
         return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())

      image_id = int(db_cursor.fetchone()[0])
             
      # - 3) alter the image uri -
      image_uri = str(image_id) + "." + image_type

      f_pgsql_execute("UPDATE images SET uri='%s' WHERE id=%d" % (image_uri,image_id))
      if db_cursor.rowcount != 1:
         return http.Response(500, {'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())
                     
      # - callback function to be called AFTER stream has been SAVED -
      def _after_save():

         # - update image state to "uploaded", in FRAMES_TO_PROCESS_DIR, and ready to be processed -
         f_pgsql_execute("UPDATE images SET state=%d WHERE id=%d" % (IMAGE_state_uploaded,image_id))
         if db_cursor.rowcount != 1:
            return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())

         # - commit database -
         f_pgsql_commit()

      # save image into FRAMES_TO_PROCESS_DIR folder
      filepath = os.path.join(FRAMES_TO_PROCESS_DIR,image_uri)
      self.file.save_stream(filepath,request.stream,_after_save)

      # - commit database -
      f_pgsql_commit()

      ret_string  = "File '%s' was scheduled for processing.\n" % image_uri
      log_msg(ret_string)
      return http.Response(201, {'Content-Type':http_headers.MimeType('text','plain'),'Location':image_uri},ret_string)

   ## ---------------------------------------------------------------------------
   # Return image identified by URI
   # @param
   def http_GET(self,request):
      log_msg("ImageFileResource::http_GET")

      # - get URI -
      uri = request.args.get("uri",None)
      if uri == None:
         ret_string = "Parameter 'uri' expected.\n"
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - get thumb indicator -
      thumb = request.args.get("thumb",None)

      # - perform GET operation based on image state -
      f_pgsql_execute("SELECT state FROM images WHERE uri='%s'" % uri[0])
      if db_cursor.rowcount != 0:
         assert db_cursor.rowcount == 1
         state = int(db_cursor.fetchone()[0])

         if state == IMAGE_state_inserted:
            ret_string = "Image specified by uri '%s' was inserted to database, but not uploaded yet.\n" % uri[0]
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         elif state == IMAGE_state_uploaded:
            if thumb == None:
               image_file_name = os.path.join(FRAMES_TO_PROCESS_DIR,uri[0])
               image_file_type = (os.path.splitext(uri[0])[-1])[1:]
            else:
               ret_string = "Image specified by uri '%s' has not generated thumbnail yet" % uri[0]
               err_msg(ret_string)
               return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         elif state == IMAGE_state_in_process or state == IMAGE_state_processed:
            if thumb == None:
               image_file_name = os.path.join(FRAMES_DIR,"%s.%s" % (uri[0],FRAME_EXT))
            else:
               image_file_name = os.path.join(FRAMES_THUMBS_DIR,"%s.%s" % (uri[0],FRAME_EXT))

            image_file_type = FRAME_EXT

         elif state == IMAGE_state_invalid:
            ret_string = "Some errors was occured while processing image specified by uri '%s'.\n" % uri[0]
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         else:
            assert 0

      else:
         ret_string = "Image specified by uri '%s' does not exist.\n" % uri[0]
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # - retrieve and return image file -
      image_file = self.file.load_stream(image_file_name)
      if image_file == None:
         ret_string = "Cannot open %s file '%s'.\n" % ("image" if thumb == None else "thumbnail",image_file_name)
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      ret_string = "Sending data of image file '%s'.\n" % image_file_name
      log_msg(ret_string)
      return http.Response(200,{'Content-Type':http_headers.MimeType('image',image_file_type)},image_file)


   ## ---------------------------------------------------------------------------
   # 
   # @param
   def http_DELETE(self, request):
      log_msg("ImageFileResource::http_DELETE")
      
      # get URI
      uri = request.args.get("uri", None)
      if uri == None:
         ret_string = "Parameter 'uri' expected.\n"
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - perform DELETE operation based on image state -
      files_to_delete = []

      f_pgsql_execute("SELECT id,state FROM images WHERE uri='%s'" % uri[0])
      if db_cursor.rowcount != 0:
         assert db_cursor.rowcount == 1
         query_result = db_cursor.fetchone()

         image_id = int(query_result[0])
         image_state = int(query_result[1])

         if image_state == IMAGE_state_uploaded:
            
            # - remove image from database -
            f_pgsql_execute("DELETE FROM images WHERE id=%d" % image_id)

            # - commit DB changes -
            f_pgsql_commit()

            # - mark file to delete -
            files_to_delete.append(os.path.join(FRAMES_TO_PROCESS_DIR,uri[0]))

         elif image_state == IMAGE_state_inserted or image_state == IMAGE_state_in_process:
            ret_string = "Image specified by uri '%s' is processed, cannot be deleted now.\n" % uri[0]
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         elif image_state == IMAGE_state_processed or image_state == IMAGE_state_invalid:

            # - remove image objects, image segments and image itself from database -
            f_pgsql_execute("DELETE FROM objects WHERE img_id=%d" % image_id)
            f_pgsql_execute("DELETE FROM segments WHERE img_id=%d" % image_id)
            f_pgsql_execute("DELETE FROM images WHERE id=%d" % image_id)
            f_pgsql_commit()

            # - mark files to delete -
            files_to_delete.append(os.path.join(FRAMES_DIR,"%s.%s" % (uri[0],FRAME_EXT)))
            files_to_delete.append(os.path.join(FRAMES_THUMBS_DIR,"%s.%s" % (uri[0],FRAME_EXT)))

            if (image_state == IMAGE_state_invalid):
               files_to_delete.append(os.path.join(FRAMES_TO_PROCESS_DIR,uri[0]))

      else:
         ret_string = "Image specified by uri '%s' does not exist.\n" % uri[0]
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # - if any file was marked to be deleted -
      if files_to_delete != []:
         
         for ftd in files_to_delete:
            f_execute("rm -rf '%s'" % ftd);

         ret_string = "image files %s has been successfully deleted.\n" % str(files_to_delete)
         log_msg(ret_string)
         return http.Response(200,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

## =============================================================================
# Class handling video files.
#
class VideoFileResource(resource.Resource):

   def allowedMethods(self):
      return ('PUT','GET','DELETE')

   ## ---------------------------------------------------------------------------
   # Constructor
   # @param f
   def __init__ (self,f):
      self.file = f

   ## ---------------------------------------------------------------------------
   # Upload video (generate unique URI).
   # @param request 
   def http_PUT(self,request):
      log_msg("VideoFileResource::http_PUT")

      # - check content type -
      contentType = request.headers.getHeader('Content-Type')
      contentType = str(contentType)
      if not contentType.startswith("MimeType('video'"):
         ret_string = "Wrong Content-Type, accepts only 'video' MIME type.\n"
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # get parameter frame_step
      frame_step = request.args.get("frame_step",None)
      if frame_step == None:
         i_frame_step = -1
      else:
         i_frame_step = int(frame_step[0])

      # get parameter exp_time - expected format: yyyy-mm-dd,hh:mm:ss
      exp_time = request.args.get("exp_time",None)
      if exp_time == None:
         s_exp_time = "NULL"
      else:
         try:
            time.strptime(exp_time[0],'%Y-%m-%d,%H:%M:%S') 
         except:
            ret_string = "Wrong format of parameter 'exp_time', expected format 'yyyy-mm-dd,hh:mm:ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

         s_exp_time = "'%s'" % exp_time[0].replace(","," ")

      # get parameter exp_loc - expected format: "(d)dd.dd dd.dd" (lat lon)
      exp_loc = request.args.get("exp_loc",None)
      if exp_loc == None:
         s_exp_loc = "NULL"
      else:
         re_subs = exp_loc_re.findall(exp_loc[0])
         if len(re_subs) == 1 and re_subs[0] == exp_loc[0]:
            # swap lattitude and longitude, PostGIS uses different ordering
            s_exp_loc = DBin_geo_point(exp_loc[0])
            s_exp_loc = "ST_GeographyFromText('SRID=4326; %s')" % s_exp_loc
         else:
            ret_string = "Wrong format of parameter 'exp_loc', expected format '[N|S]dd.dd:mm.mm:ss.ss,[E|W]ddd.dd:mm.mm:ss.ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - extract image type -
      video_type = contentType[len("MimeType('video', '"):]
      video_type = video_type[:video_type.find("'")]

      # - 1) insert video with void uri and state set to "inserted" in DB, not uploaded yet -
      f_pgsql_execute("INSERT INTO videos (id,uri,exp_time,exp_loc,frame_step,img_cnt,done_img_cnt,state) VALUES (DEFAULT,'',%s,%s,%d,0,0,%d)" % (s_exp_time,s_exp_loc,i_frame_step,VIDEO_state_inserted))
      if db_cursor.rowcount != 1:
         return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())

      # - 2) retrieve video id -
      f_pgsql_execute("SELECT MAX(id) FROM videos")
      if db_cursor.rowcount != 1:
         return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())

      video_id = int(db_cursor.fetchone()[0])

      # - 3) alter the video uri -
      video_uri = "%d.%s" % (video_id,video_type)

      f_pgsql_execute("UPDATE videos SET uri='%s' WHERE id=%d" % (video_uri,video_id))
      if db_cursor.rowcount != 1:
         return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())

      # - callback function to be called AFTER stream has been SAVED -
      def _after_save():

         # - update video state to "uploaded", in VIDEOS_DIR, and ready to be processed -
         f_pgsql_execute("UPDATE videos SET state=%d WHERE id=%d" % (VIDEO_state_uploaded,video_id))
         if db_cursor.rowcount != 1:
            return http.Response(500,{'Content-Type':http_headers.MimeType('text','plain')},f_pgsql_database_error())

         # - commit database -
         f_pgsql_commit()

      # save image into FRAMES_TO_PROCESS_DIR folder
      filepath = os.path.join(VIDEOS_DIR,video_uri)
      self.file.save_stream(filepath,request.stream,_after_save)

      # - commit database -
      f_pgsql_commit()

      ret_string = "Video file '%s' was scheduled for processing\n" % video_uri
      log_msg(ret_string)
      return http.Response(201,{'Content-Type':http_headers.MimeType('text','plain'),'Location':video_uri},ret_string)

   ## ---------------------------------------------------------------------------
   # Return video identified by URI
   # @param
   def http_GET(self, request):
      log_msg("VideoFileResource::http_GET")

      # get URI
      uri = request.args.get("uri",None)
      if uri == None:
         ret_string = "Parameter 'uri' expected.\n"
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - get thumb indicator -
      thumb = request.args.get("thumb",None)

      # - perform GET operation based on video state -
      f_pgsql_execute("SELECT state FROM videos WHERE uri='%s'" % uri[0])
      if db_cursor.rowcount != 0:
         assert db_cursor.rowcount == 1
         video_state = int(db_cursor.fetchone()[0])

         if video_state == VIDEO_state_inserted:
            ret_string = "Video specified by uri '%s' was inserted to database, but not uploaded yet.\n" % uri[0]
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         elif video_state == VIDEO_state_uploaded or video_state == VIDEO_state_in_process:
            send_file_name = os.path.join(VIDEOS_DIR,uri[0])
            send_file_type = (os.path.splitext(uri[0])[-1])[1:]

         elif video_state == VIDEO_state_processed:
            if thumb == None:
               send_file_name = os.path.join(VIDEOS_DIR,uri[0])
               send_file_type = (os.path.splitext(uri[0])[-1])[1:]
            else:
               send_file_name = os.path.join(VIDEOS_THUMBS_DIR,"%s.%s" % (uri[0],FRAME_EXT))
               send_file_type = FRAME_EXT

         elif video_state == VIDEO_state_invalid:
            ret_string = "Some errors was occured while processing video specified by uri '%s'.\n" % uri[0]
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         else:
            assert 0

      else:
         ret_string = "Video specified by uri '%s' does not exist.\n" % uri[0]
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # - retrieve and return video file -
      video_file = self.file.load_stream(send_file_name)
      if video_file == None:
         ret_string = "Cannot open %s file '%s'.\n" % ("video" if thumb == None else "thumbnail",send_file_name)
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      ret_string = "Sending data of %s file '%s'.\n" % ("video" if thumb == None else "image",send_file_name)
      log_msg(ret_string)
      return http.Response(200,{'Content-Type':http_headers.MimeType("video" if thumb == None else "image",send_file_type)},video_file)

   ## ---------------------------------------------------------------------------
   # 
   # @param
   def http_DELETE(self,request):
      log_msg("VideoFileResource::http_DELETE")
      
      # get URI
      uri = request.args.get("uri",None)
      if uri == None:
         ret_string = "Parameter 'uri' expected.\n"
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - perform DELETE operation based on video state -
      files_to_delete = []

      f_pgsql_execute("SELECT id,state FROM videos WHERE uri='%s'" % uri[0])
      if db_cursor.rowcount != 0:
         assert db_cursor.rowcount == 1
         query_result = db_cursor.fetchone()

         video_id = int(query_result[0])
         video_state = int(query_result[1])

         if video_state == VIDEO_state_uploaded:
            
            # - remove video from database -
            f_pgsql_execute("DELETE FROM videos WHERE id=%d" % video_id)
            f_pgsql_commit()

            # - mark file to delete -
            files_to_delete.append(os.path.join(VIDEOS_DIR,uri[0]))

         elif video_state == VIDEO_state_inserted or video_state == VIDEO_state_in_process:
            ret_string = "Video specified by uri '%s' is processed, cannot be deleted now.\n" % uri[0]
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         elif video_state == VIDEO_state_processed or video_state == VIDEO_state_invalid:
            
            # - delete video, all its images, and their segments and objects -
            f_pgsql_execute("SELECT id,uri FROM images WHERE video_id=%d" % video_id)
            if db_cursor.rowcount != 0:
               v_images_result = db_cursor.fetchall()
               for v_image in v_images_result:
                  image_id = int(v_image[0])
                  image_uri = v_image[1]

                  f_pgsql_execute("DELETE FROM objects WHERE img_id=%d" % image_id)
                  f_pgsql_execute("DELETE FROM segments WHERE img_id=%d" % image_id)
                  f_pgsql_execute("DELETE FROM images WHERE id=%d" % image_id)

                  # - mark files to delete -
                  files_to_delete.append(os.path.join(FRAMES_DIR,"%s.%s" % (image_uri,FRAME_EXT)))
                  files_to_delete.append(os.path.join(FRAMES_THUMBS_DIR,"%s.%s" % (image_uri,FRAME_EXT)))
                  files_to_delete.append(os.path.join(FRAMES_TO_PROCESS_DIR,image_uri))

            f_pgsql_execute("DELETE FROM videos WHERE id=%d" % video_id)
            files_to_delete.append(os.path.join(VIDEOS_DIR,uri[0]))
            files_to_delete.append(os.path.join(VIDEOS_THUMBS_DIR,"%s.%s" % (uri[0],FRAME_EXT)))

            # - commit DB changes -
            f_pgsql_commit()

      else:
         ret_string = "Video specified by uri '%s' does not exist.\n" % uri[0]
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # - if any file was marked to be deleted -
      if files_to_delete != []:
         
         for ftd in files_to_delete:
            f_execute("rm -rf '%s'" % ftd);

         ret_string = "Video and image files %s has been successfully deleted.\n" % str(files_to_delete)
         log_msg(ret_string)
         return http.Response(200,{'Content-Type':http_headers.MimeType('text','plain')}, ret_string)

## =============================================================================
# Class handling queries.
#
class QueryResource(resource.Resource):

   def allowedMethods(self):
      return ('GET')

   ## ---------------------------------------------------------------------------
   # Constructor
   # @param f
   def __init__ (self, f):
      self.file = f

   def get_exp_time_arguments(self,request):
      #{{{
      # - check exp_time_from and exp_time_to parameters -
      s_exp_time_from = request.args.get("exp_time_from",None)
      s_exp_time_to   = request.args.get("exp_time_to"  ,None)
      exp_time = None

      # is one of the parameters missing?
      if s_exp_time_from != None and s_exp_time_to == None:
         ret_string = "Parameter 'exp_time_to' expected."
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      if s_exp_time_from == None and s_exp_time_to != None:
         ret_string = "Parameter 'exp_time_from' expected."
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)
      
      # both parameters specified, check their formats
      if s_exp_time_from != None and s_exp_time_to != None:
         try:
            time.strptime(s_exp_time_from[0],'%Y-%m-%d,%H:%M:%S') 
         except:
            ret_string = "Wrong format of parameter 'exp_time_from', expected format 'yyyy-mm-dd,hh:mm:ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         try:
            time.strptime(s_exp_time_to[0],'%Y-%m-%d,%H:%M:%S') 
         except:
            ret_string = "Wrong format of parameter 'exp_time_to', expected format 'yyyy-mm-dd,hh:mm:ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

         exp_time_from = s_exp_time_from[0].replace(","," ")
         exp_time_to = s_exp_time_to[0].replace(","," ")

         exp_time = (exp_time_from,exp_time_to)

      return exp_time
      #}}}
      

   def get_exp_loc_arguments(self,request):
      #{{{
      # - check exp_time_from and exp_time_to parameters -
      s_exp_loc_center = request.args.get("exp_loc",None)
      s_radius = request.args.get("radius" ,None)
      exp_loc = None

      # is one of the parameters missing?
      if s_exp_loc_center != None and s_radius == None:
         ret_string = "Parameter 'radius' expected."
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      if s_exp_loc_center == None and s_radius != None:
         ret_string = "Parameter 'exp_loc' expected."
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)
      
      # both parameters specified, check their formats
      if s_exp_loc_center != None and s_radius != None:
         re_subs = exp_loc_re.findall(s_exp_loc_center[0])
         if len(re_subs) == 1 and re_subs[0] == s_exp_loc_center[0]:
            exp_loc_center = s_exp_loc_center[0]
         else:
            ret_string = "Wrong format of parameter 'exp_loc', expected format '[N|S]dd.dd:mm.mm:ss.ss,[E|W]ddd.dd:mm.mm:ss.ss'.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)
         
         re_subs = radius_re.findall(s_radius[0])
         if len(re_subs) == 1 and re_subs[0][0] == s_radius[0]:
            radius = s_radius[0]
         else:
            ret_string = "Wrong format of parameter 'radius', expected format is float number.\n"
            err_msg(ret_string)
            return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)
      
         exp_loc = (exp_loc_center,radius)

      return exp_loc
      #}}}

   ## ---------------------------------------------------------------------------
   # Return image identified by URI
   # @param
   def http_GET(self, request):
      log_msg("QueryResource::http_GET")

      # get query_type
      query_type = request.args.get("q",None)
      if query_type == None:
         ret_string = "Parameter 'q' expected."
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

      # - test query type validity -
      if not (query_type[0] in ["getResourceList","getResourceState","getSegments","getObjects","isObjectInSegment","isSegmentAdjacent","tag"]):
         ret_string = "Unrecognized query type q=%s." % query_type[0]
         err_msg(ret_string)
         return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)

      # - test query type -
      if query_type[0] == "getResourceList":
      
         exp_time = self.get_exp_time_arguments(request)
         if exp_time != None and type(exp_time) != tuple:
            return exp_time
        
         exp_loc = self.get_exp_loc_arguments(request)
         if exp_loc != None and type(exp_loc) != tuple:
            return exp_loc
         
         # call the query
         ret_string = f_query_getResourceList(exp_time,exp_loc)

      else:

         # - retrieve resource_uri -
         resource_uri = request.args.get("resource_uri",None)

         if query_type[0] in ["getResourceState","getSegments","getObjects"]:
            
            if resource_uri == None:
               ret_string = "Parameter 'resource_uri' expected."
               err_msg(ret_string)
               return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

            resource_uri = resource_uri[0]

            if query_type[0] == "getResourceState":
               ret_string = f_query_getResourceState(resource_uri)

            elif query_type[0] == "getSegments":
               ret_string = f_query_getSegments(resource_uri)

            elif query_type[0] == "getObjects":
               ret_string = f_query_getObjects(resource_uri)

         else:

            if resource_uri != None:
               resource_uri = resource_uri[0]

            if query_type[0] == "isObjectInSegment":
               
               # - retrieve segment_type -
               segment_type = request.args.get("segment_type",None)
               if segment_type == None:
                  ret_string = "Query \"%s\" expects parameter 'segment_type'." % query_type[0]
                  err_msg(ret_string)
                  return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

               # - retrieve object_type -
               object_type = request.args.get("object_type",None)
               if object_type == None:
                  ret_string = "Query \"%s\" expects parameter 'object_type'." % query_type[0]
                  err_msg(ret_string)
                  return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)
                  
               exp_time = self.get_exp_time_arguments(request)
               if exp_time != None and type(exp_time) != tuple:
                  return exp_time
               
               exp_loc = self.get_exp_loc_arguments(request)
               if exp_loc != None and type(exp_loc) != tuple:
                  return exp_loc
               
               # call the query
               ret_string = f_query_isObjectInSegment(resource_uri,segment_type[0],object_type[0], exp_time, exp_loc)

            elif query_type[0] == "isSegmentAdjacent":

               # - retrieve segment_type1 -
               segment_type1 = request.args.get("segment_type1",None)
               if segment_type1 == None:
                  ret_string = "Query \"%s\" expects parameter 'segment_type1'." % query_type[0]
                  err_msg(ret_string)
                  return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

               # - retrieve segment_type2 -
               segment_type2 = request.args.get("segment_type2",None)
               if segment_type2 == None:
                  ret_string = "Query \"%s\" expects parameter 'segment_type2'." % query_type[0]
                  err_msg(ret_string)
                  return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)
               
               exp_time = self.get_exp_time_arguments(request)
               if exp_time != None and type(exp_time) != tuple:
                  return exp_time
               
               exp_loc = self.get_exp_loc_arguments(request)
               if exp_loc != None and type(exp_loc) != tuple:
                  return exp_loc
               
               # call the query
               ret_string = f_query_isSegmentAdjacent(resource_uri,segment_type1[0],segment_type2[0],exp_time,exp_loc)

            elif query_type[0] == "tag":

               # - retrieve tag -
               tag = request.args.get("tag",None)
               if tag == None:
                  ret_string = "Query \"%s\" expects parameter 'tag'." % query_type[0]
                  err_msg(ret_string)
                  return http.Response(400,{'Content-Type':http_headers.MimeType('text','plain')},ret_string)	

               exp_time = self.get_exp_time_arguments(request)
               if exp_time != None and type(exp_time) != tuple:
                  return exp_time

               exp_loc = self.get_exp_loc_arguments(request)
               if exp_loc != None and type(exp_loc) != tuple:
                  return exp_loc

               ret_string = f_query_tag(resource_uri,tag[0],exp_time,exp_loc)

      return http.Response(200,{'Content-Type':http_headers.MimeType('application','xml')},ret_string)

# Create resources #############################################################

f = File()

root = static.File("root")
root.putChild("example",ExampleResource()) 
root.putChild("image",ImageFileResource(f)) 
root.putChild("video",VideoFileResource(f)) 
root.putChild("query",QueryResource(f)) 

root = log.LogWrapperResource(root)
log.DefaultCommonAccessLoggingObserver().start()

site = server.Site(root)
application = service.Application('web')

service = strports.service('tcp:' + SERVER_PORT, channel.HTTPFactory(site))
service.setServiceParent(application)

#TODO: odpojovat DB pri shutdownu? mozna pomoci reactoru http://twistedmatrix.com/pipermail/twisted-python/2006-April/012888.html

