<?php

class DBException extends Exception 
{
	public $backtrace;
	
	public function __construct($message, $code = 0)
	{
		$this->message = $message. " Code: $code";
		$this->code = $code;
		$this->backtrace = debug_backtrace();
	}
}

class DBExceptionNoConnect extends DBException 
{
	
}

class DBExceptionNoSelect extends DBException 
{
	
}

abstract class DBExceptionQueryErrorAbstract extends DBException 
{
	private $errorN;

	public function getNumberError()
	{
		return $this->errorN;
	}
	public abstract function isDuplicateKey();
	public abstract function isTableExist();
	public abstract function isFunctionExist();
}


interface IDBPrototype 
{
	/* KONSTANTY*/
	
	/**
	 * vysldek jako objekt
 	 */	
	const OBJECT = 1;
	
	/**
 	 * vysledek jako asocativni pole
 	 */	
	const ARRAY_A = 2;
	
	/**
 	 * vysledek jako ciselne pole
     */
	const ARRAY_N = 4;
	
	/**
	 * otevreni databaze
	 *
	 * @param string $dbhost
	 * @param string $dbuser
	 * @param string $dbpassword
	 * @param string $dbname
	 * @param string $character
	 */
	public function init($dbhost, $dbuser, $dbpassword, $dbname, $character = "utf8");
	/**
	 * Metoda nastavi kodovani vystupnich dat z dotazu. Pokud to DB dovoluje
	 *
	 * @param string $character_set kod kodovani
	 */
	public function setCharacterResult($character_set = 'utf8', $collation = 'utf8_czech_ci');
	
	/**
	 * Vybere ze kterou DB se bude pracovat
	 * 
	 * @param string $dbName - jmeno databaze
	 */
	public function select($dbName);
	

	/**
	 * metoda vytvori predpripraveny (ulozeny) dotaz. Tj. do dotazu misto hodnot se vklada symbol '?'
	 * po teto metode se provede prikaz execute. 
	 * 
	 * @var string sql - SQL dotaz kde jsou promnene nahrazeny otaznikem 
	 * 
	 * @example 
	 * <code>
	 * 	$db->prepare("SELECT jmeno, prijmeni, prava FROM uzivatel WHERE id = ?");
	 * </code>
	 *
	 * @param string $query
	 */
	public function prepare($query);


	/**
	 * Metoda vlozi data za otaznik do ulozeneho dotazu a pak tento prikaz provede.
	 * Metoda funguje stejne jako metoda query a vraci i stejny vysledek
	 * 
	 * @var list of type parametrs
	 *
	 * @return int pripade dotazu delete|update|insert|replace vrati pocet ovlivnenych radku
	 *             pri selectu vrati pocet vybranych radku
	 */
	public function execute($mixedArray);
	
	
	
	/**
	 * funkce zpracuje dotaz a ulozi vysledek do Cache
	 * 
	 * @param string $query Dotaz ktery chcete provest
	 * 
	 * @return int pripade dotazu delete|update|insert|replace vrati pocet ovlivnenych radku
	 *             pri selectu vrati pocet vybranych radku
	 */
	public function query($query);

	/**
	 * Funkce vrati skalarni (jednu) hodnotu
	 * 
	 * @param string $query [nepovynny parametr]Dotaz  
	 * @param int $x [default 0] hodnota. cislo sloupce  
	 * @param int $y [default 0] hodnota. cislo radku 
	 */
	public function getVar($query = null, $x = 0, $y = 0);
	
		
    /**
     * Vrati radek ze zanamu.
     * 
     * @param string $query [nepovinny parametr] Dotaz
     * @param const $output [default OBJECT] typ vracenych dat
     * @param int $y cislo radu ktery se ma vratit
     * 
     */
    public function getRow($query = null, $output = IDBPrototype::OBJECT, $y = 0);
   
    /**
     * Vrati sloupec ze zanamu.
     * 
     * @param string $query [nepovinny parametr] Dotaz
     * @param int $x cislo slopce ktery se ma vratit
     * 
     */
    public function getCol($query = null, $x = 0);
   

	/**
     * Vrati vysledek dotazu
     * 
     * @param string $query [nepovinny parametr] Dotaz
     * @param const $output [default OBJECT] typ vracenych dat
     * 
     */
	public function getResults($query = null, $output = IDBPrototype::OBJECT);
	
	
	/**
	 * Vrati data z dotazu v podobe objektu ResultSet
	 *
	 * @param enum(ARRAY_A|ARRAY_N) $output typ indexu jakym se budou indexovat jednotlive parametry
	 * 		  (cislem | jmenem) 
	 * @return ResultSet Object 
	 */
	public function getResulSet($output =  IDBPrototype::ARRAY_A);
	
	
	/**
	 * Vrati informace o slopecku 
	 * 
	 */
	public function getColInfo($info_type = "name", $col_offset = -1);
	
						
	/**
	 * Vrati pocet radku po dotazu select
	 * 
	 * @return pocet radku nebo -1 
	 */
	public function getNumRows();
	
    /**
     * Vrati ID posledniho INSERT|REPLACE dotazu
     * 
     */
    public function getLastID();
        
    /**
     * Metoda vrati retezec, tteri identifikuje jmeno driveru
     * @return string
     *
     */
    public function getDriverName();
    
    /**
     * Vrati verzi driveru 
     * @return float
     */
	public function getVersion();

	/**
	 * Zjisti jestli existuje tabulka 
	 *
	 * @param string $tableName jmneo tabulky 
	 * @return  bool true pokud tabulka existuje
	 */
	public function tableExist($tableName);
	
    /**
     * Zacatek transakce (if supported).
     * @return void
     */
    public function begin();

    /**
     * potvrzeni transakce
     * @return void
     */
    public function commit();

    /**
     * Zruseni transakce
     * @return void
     */
    public function rollback();
}
?>
