
from twisted.web2 import server, http, resource, channel, static, http_headers, log
from twisted.internet import reactor
from twisted.application import service, internet, strports

from twisted.web2.stream import readStream

import sys
import os.path
import os
import threading
import subprocess
import Queue
import settings
import re
import base64

def normalize_name(n):
    """ recognizer cuts the .wav extension if present, so just cut it before so we don't have surprises later, also the name must not contain underscore """
    if n.endswith(".wav"):
        n = n[:-len(".wav")]

    n = n.replace("_", "-")

    return n

def query(q):
    p = subprocess.Popen([os.path.join(settings.indexer_path, "scripts", "8_search.sh"), q], cwd=os.path.join(settings.indexer_path, "scripts"), stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout, stderr = p.communicate()
    return stdout

class Request:
    pass

class RequestQuit(Request):
    pass

class RequestRecognition(Request):
    def __init__ (self, name):
        self.name = name

class RequestIndexing(Request):
    pass

class ProcessingThread(threading.Thread):
    def __init__ (self, q):
        threading.Thread.__init__ (self)
        self.queue = q
        self.setDaemon(True)

    def process_recognition(self, request):
        if not os.path.exists(settings.lat_path):
            os.makedirs(settings.lat_path)

        # delete existing lattices for the file
        r = re.compile(r"(.*)_([0-9]+)_([0-9]+).lat")
        for filename in os.listdir(settings.lat_path):
            M = r.match(filename)
            if M is not None:
                if M.group(1) == request.name:
                    os.remove(os.path.join(settings.lat_path, filename))

        sys.stderr.write("starting recognizer for %s\n" % (request.name))
        p = subprocess.Popen([os.path.join(settings.recognizer_path, "offlinerec"), "-i", os.path.join(settings.wav_path, request.name), "-d", settings.lat_path, "-o", "/dev/null"], cwd=settings.recognizer_path, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        stdout, stderr = p.communicate()

        n_lats = 0
        for filename in os.listdir(settings.lat_path):
            M = r.match(filename)
            if M is not None:
                if M.group(1) == request.name:
                    n_lats += 1

        if n_lats == 0:
            sys.stderr.write("recognizer for %s did not generate any lattices! \nstdout: %s\nstderr: %s\n" % (request.name, stdout, stderr))
        else:
            sys.stderr.write("recongizer for %s ends. Generated %d lattice files\n" % (request.name, n_lats))

    def process_indexing(self, request):
        sys.stderr.write("starting indexer\n")    
        p = subprocess.Popen([os.path.join(settings.indexer_path, "run.sh")], cwd=settings.indexer_path, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        stdout, stderr = p.communicate()
        sys.stderr.write("indexer ended\n")

    def run(self):
        while True:
            request = self.queue.get()
            if isinstance(request, RequestQuit):
                break
            elif isinstance(request, RequestRecognition):
                self.process_recognition(request)
            elif isinstance(request, RequestIndexing):
                self.process_indexing(request)


class Db:
    def __init__ (self, root):
        self.root = root
        if not os.path.exists(root):
            os.makedirs(root)

    def exists(self, name):
        return os.path.exists(os.path.join(self.root, name))

    def put(self, q, name, stream):
        f = open(os.path.join(self.root, name), "w")
        
        def gotData(newdata):
            f.write(newdata)

        d = readStream(stream, gotData)

        def _finishedReading(ignore):
            f.close()
            q.put(RequestRecognition(name))

        d.addCallback(_finishedReading)

    def delete(self, name):
        os.remove(os.path.join(self.root, name))

    def get(self, name):
        f = open(os.path.join(self.root, name), "r")
        return f

    def list(self):
        return os.listdir(self.root)

class WavFileResource(resource.Resource):

    def allowedMethods(self):
        return ('GET', 'PUT', 'DELETE')

    def __init__ (self, db, q):
        self.db = db
        self.queue = q

    def locateChild(self, ctx, segments):
          return self, ()

    def http_GET(self, request):
        split = request.path.split("/") 
        if len(split) != 3:
            return http.Response(403, {'Content-Type': http_headers.MimeType('text', 'plain')}, stream="File name must not contain slashes.\n")

        filename = normalize_name(split[2])
        f = self.db.get(filename)

        return http.Response(200, {'Content-Type':http_headers.MimeType('audio', 'x-wav')}, stream=f)

    def http_PUT(self, request):
        split = request.path.split("/") 
        if len(split) != 3:
            return http.Response(403, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="File name must not contain slashes.\n")

        filename = normalize_name(split[2])

        contentType = request.headers.getHeader('Content-Type')
        if contentType != "audio/x-wav":
            return http.Response(403, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="Wrong Content-Type, accepts only " + self.contentType +"\n")


        self.db.put(self.queue, filename, request.stream)

        return http.Response(201, {'Content-Type':http_headers.MimeType('text', 'plain')}, "File '" + filename + "' stored.\n")

    def http_DELETE(self, request):
        split = request.path.split("/") 
        if len(split) != 3:
            return http.Response(403, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="File name must not contain slashes.\n")

        filename = normalize_name(split[2])

        if not self.db.exists(filename):
            return http.Response(403, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="File does not exist.\n")
 

        self.db.delete(filename)

        return http.Response(200, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="File '" + filename + "' has been deleted.\n")

class PreprocessResource(resource.Resource):
    
    def allowedMethods(self):
        return ('GET',)

    def __init__ (self, q):
        self.queue = q
    
    def http_GET(self, request):

        self.queue.put(RequestIndexing())

        return http.Response(202, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="Preprocessing has started.\n")


class QueryResource(resource.Resource):

    def allowedMethods(self):
        return ('GET',)


    def http_GET(self, request):
        qs = request.args.get("q", [])
        if len(qs) == 0:
            return http.Response(403, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="You must specify 'q' parameter.\n")

        q = qs[0]
        result = query(q)
        result = '<?xml version="1.0"?>\n' + result

        return http.Response(200, {'Content-Type':http_headers.MimeType('application', 'xml')}, stream=result)

class ListResource(resource.Resource):
    def __init__ (self, db):
        self.db = db
 
    def allowedMethods(self):
        return ('GET',)

    def http_GET(self, request):
        ret = ""
        for filename in self.db.list():
            uri = filename
            uri = uri.replace("(", "_")
            uri = base64.b64decode(uri)
            ret += uri + "\t" + filename + "\n"

        return http.Response(200, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream=ret)


class LatticeResource(resource.Resource):
    def __init__ (self, lat_path):
        self.lat_path = lat_path

    def allowedMethods(self):
        return ('GET',)

    def locateChild(self, ctx, segments):
          return self, ()

    def http_GET(self, request):

        split = request.path.split("/") 
        if len(split) != 3:
            return http.Response(403, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream="File name must not contain slashes.\n")

        name = split[2]

        lattices = []

        r = re.compile(r"(.*)_([0-9]+)_([0-9]+).lat")
        for filename in os.listdir(self.lat_path):
            M = r.match(filename)
            if M is not None:
                if M.group(1) == name:
                    lattices.append (filename)

        lattices.sort()

        response = ""
        for lattice in lattices:
            fn = os.path.join(self.lat_path, lattice)
            f = open (fn, "r")
            response += "# " + lattice + "\n"
            response += f.read()
            f.close()

        return http.Response(200, {'Content-Type':http_headers.MimeType('text', 'plain')}, stream=response)


db = Db(settings.wav_path)

# processing queue for requests (recognizer and indexing)
q = Queue.Queue()
pt = ProcessingThread(q)
pt.start()

wavFileResource = WavFileResource(db, q)

root = static.File("root")
root.putChild("wav", wavFileResource) 
root.putChild("preprocess", PreprocessResource(q))
root.putChild("query", QueryResource())
root.putChild("list", ListResource(db))
root.putChild("lattice", LatticeResource(settings.lat_path))

root = log.LogWrapperResource(root)
log.DefaultCommonAccessLoggingObserver().start()



site = server.Site(root)

application = service.Application('web')
s = strports.service('tcp:9080', channel.HTTPFactory(site))
s.setServiceParent(application)



