﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using ObjectModel.InformationElements.Renderers;
using ObjectModel.Protocols;

namespace AnalyzeThisForms.Components
{
	public delegate void UnloadPravidlaEventHandler();
	[Serializable]
	public class ProgramInfo
	{
		string pravidlaFileName = "Pravidla.dll";
		Dictionary<string, Type> types = new Dictionary<string, Type>();
		Dictionary<string, Type> loaderTypes = new Dictionary<string, Type>();
		Dictionary<string, Type> protocolFactoryTypes = new Dictionary<string, Type>();
		Dictionary<string, Type> outputPlugins = new Dictionary<string, Type>();
		ProgramSettings settings;
		public event UnloadPravidlaEventHandler UnloadPravidlaEvent;
		public event UnloadPravidlaEventHandler PravidlaLoadedEvent;
		AppDomain domain;

		public ProgramInfo()
		{ }

		public string PravidlaFileName
		{
			get { return this.pravidlaFileName; }
			set { this.pravidlaFileName = value; }
		}
		public Dictionary<string, Type> Types
		{
			get { return this.types; }
		}
		public ProgramSettings Settings
		{
			get { return settings; }
			set { settings = value; }
		}
		public Dictionary<string, Type> OutputPlugins
		{
			get { return outputPlugins; }
			set { outputPlugins = value; }
		}
		public Dictionary<string, Type> LoaderTypes
		{
			get { return loaderTypes; }
			set { loaderTypes = value; }
		}

		public void LoadPravidla(string dir)
		{
            string pravidlaPath = Path.Combine(dir, pravidlaFileName);
            if (!File.Exists(pravidlaPath))
            {
                return;
            }
			byte[] assemblyBytes = File.ReadAllBytes(pravidlaPath);			
			Assembly assembly = Assembly.Load(assemblyBytes);
			this.types.Clear();

			foreach (Type type in  assembly.GetTypes())
			{
				if (!type.IsClass || type.IsNotPublic) continue;
				Type[] interfaces = type.GetInterfaces();
				if (((IList)interfaces).Contains(typeof(ObjectModel.InformationElements.IObjectbase)))
				{
					this.types.Add(type.FullName, type);
				}
			}
			if (this.PravidlaLoadedEvent != null)
				this.PravidlaLoadedEvent();
		}
		
		public void UnloadPravidla()
		{
			this.types.Clear();
			if (this.UnloadPravidlaEvent != null)
				this.UnloadPravidlaEvent();
			GC.Collect();
		}
		public void LoadPlugins(string pluginDir)
		{

			this.loaderTypes.Clear();
			this.outputPlugins.Clear();
			this.protocolFactoryTypes.Clear();

            AppDomain.CurrentDomain.AppendPrivatePath(ProgramSettings.PluginsSubDir);
            //AppDomain.CurrentDomain.SetupInformation.PrivateBinPath = ProgramSettings.PluginsSubDir;

			foreach (string file in Directory.GetFiles(pluginDir, "*.dll"))
			{
				string fullpath = Path.Combine(pluginDir, file);                            
                Assembly assembly = Assembly.LoadFile(fullpath);
                
				foreach (Type type in assembly.GetTypes())
				{
					if (!type.IsClass || type.IsNotPublic) continue;
					if (type.IsSubclassOf(typeof(ObjectModel.Protocols.ProtocolFactoryBase)))
					{
						this.protocolFactoryTypes.Add(type.FullName, type);
					}
					else if (type.IsSubclassOf(typeof(ObjectModel.Loaders.LoaderBase)))
					{
						this.loaderTypes.Add(type.FullName, type);
					}
					else if (((IList)type.GetInterfaces()).Contains(typeof(ObjectModel.InformationElements.Renderers.IInformationElementRenderer)))
					{
						this.outputPlugins.Add(type.FullName, type);
					}
				}
			}

		}
		public void LoadSettings(string settingDir)
		{
			this.settings = ProgramSettings.Load(settingDir);
		}
		public void SaveSettings(string settingDir)
		{
			this.settings.Save(settingDir);
		}

		public Object GetObject(string typeName, ObjectModel.InformationElements.InformationElementBase parent, int dataBase, int dataOffset)
		{
			Object o = null;
			if (this.types.ContainsKey(typeName))
			{
				o = Activator.CreateInstance(types[typeName], parent, dataBase, dataOffset);
			}
			return o;
		}
		public ObjectModel.Loaders.LoaderBase GetLoader(string typeName)
		{
			Object o = null;
			if (this.loaderTypes.ContainsKey(typeName))
			{
				o = Activator.CreateInstance(loaderTypes[typeName]);
			}
			return o as ObjectModel.Loaders.LoaderBase;
		}
		public ProtocolFactoryBase GetProtocolFactory(string typeName)
		{
			Object o = null;
			if (this.protocolFactoryTypes.ContainsKey(typeName))
			{
				o = Activator.CreateInstance(protocolFactoryTypes[typeName], this);
			}
			return o as ProtocolFactoryBase;
		}
		public IInformationElementRenderer GetRenderer(string typeName)
		{
			Object o = null;
			if (this.outputPlugins.ContainsKey(typeName))
			{
				o = Activator.CreateInstance(outputPlugins[typeName]);
			}
			return o as IInformationElementRenderer;
		}
	}
}
