﻿using System;
///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Vstupni zasuvny modul poskytujici vstup dat ze spektralniho analyzatoru.
///

namespace Loaders.IQLoader
{
    /// <summary>
    /// Typ modulace. MT_NOTHING je vychozi hodnota, ktera vsak neni platna.
    /// </summary>
    public enum EModulationType { MT_NOTHING, MT_FSK, MT_PSK, MT_ASK, MT_MSK, MT_BPSK, MT_QPSK };


    /// <summary>
    /// Typ Data provideru. Bud soubor, analyzator nebo nic (pocatecni stav).
    /// </summary>
    public enum EDataProvider { DP_FILE, DP_ANALYZER, DP_NONE };


    /// <summary>
    /// Nese veskere obecne nastaveni pluginu. 
    /// </summary>
    public class PluginSettings
    {
        /// <summary>
        /// Vzorkovaci frekvence.
        /// </summary>
        private int sampleRate = 0;

        /// <summary>
        /// Informace o pouzitem protocol provideru.
        /// </summary>
        private ProviderInfo protocol = null;

        /// <summary>
        /// Informace o pouzitem filtru.
        /// </summary>
        private ProviderInfo filter = null;

        /// <summary>
        /// Informace o pouzitem demodulatoru.
        /// </summary>
        private ProviderInfo modulation = null;

        /// <summary>
        /// Nosna frekvence v Hz. Platne pouze pokud je dataProvider nastaven na DP_ANALYZER.
        /// </summary>
        private int centerFrequency = 0;

        /// <summary>
        /// Sirka pasma. Platne pouze pokud je dataProvider nastaven na DP_ANALYZER.
        /// </summary>
        private int bandwidth = 0;

        /// <summary>
        /// Resource descriptor pro VISA.
        /// Platny pouze pokud je dataProvider nastaven na DP_ANALYZER.
        /// </summary>
        private string resourceDescriptor = "";

        /// <summary>
        /// Pocet vzorku, ktere se nachytaji z analyzatoru. 
        /// Platne pouze pokud je dataProvider nastaven na DP_ANALYZER.
        /// </summary>
        private int pocetVzorku = 0;

        /// <summary>
        /// Konstruktor.
        /// </summary>
        public PluginSettings()
        {
            this.FilePath = "";
            this.RefLevel = -20;
        }

        /// <summary>
        /// Data provider. Bud soubor nebo vstup z analyzatoru zkrze jeho API.
        /// </summary>
        public EDataProvider dataProvider = EDataProvider.DP_NONE;

        /// <summary>
        /// Cesta ke zdrojovemu souboru. Platna pouze pokud je jako data provider DP_FILE.
        /// </summary>
        public string FilePath { get; set; }

        /// <summary>
        /// Reference level [dBm].
        /// </summary>
        public double RefLevel { get; set; }

        /// <summary>
        /// Vzorkovaci frekvence v Hz. Mela by byt vetsi nez nula.
        /// </summary>
        /// <exception cref="ArgumentException">Pokud je frekvence zaporna nebo nulova.</exception>
        public int SampleRate
        {
            get
            {
                return this.sampleRate;
            }
            set
            {
                if (value > 0)
                    this.sampleRate = value;
                else
                    throw new ArgumentException("Vzorkovaci frekvence musi byt vetsi nez nula.");
            }

        }

        /// <summary>
        /// Nosna frekvence v Hz. Musi byt vetsi nez nula.
        /// </summary>
        /// <exception cref="ArgumentException">Pokud je frekvence zaporna nebo nulova.</exception>
        public int CenterFrequency
        {
            get
            {
                return this.centerFrequency;
            }
            set
            {
                if (value > 0)
                    this.centerFrequency = value;
                else
                    throw new ArgumentException("Center frequency musi byt vetsi nez nula.");
            }
        }

        /// <summary>
        /// Sirka pasma v Hz. Musi byt vetsi nez nula.
        /// </summary>
        /// <exception cref="ArgumentException">Pokud je zadana nulva nebo zaporna.</exception>
        public int Bandwith
        {
            get
            {
                return this.bandwidth;
            }

            set
            {
                if (value <= 0)
                    throw new ArgumentException("Šířka pásma musí být celé kladné číslo.");
                else
                    this.bandwidth = value;
            }
        }

        /// <summary>
        /// Pouzity Protocol Provider.
        /// </summary>
        /// <exception cref="ArgumentNullException">Pokud bylo predano null.</exception>
        public ProviderInfo ProtocolProvider
        {
            get
            {
                return this.protocol;
            }
            set
            {
                if (value == null)
                    throw new ArgumentNullException("Protocol provider nemuze byt null.");
                else
                    this.protocol = value;
            }
        }

        /// <summary>
        /// Pouzity Demodulation Provider.
        /// </summary>
        /// <exception cref="ArgumentNullException">Pokud bylo predano null.</exception>
        public ProviderInfo DemodulationProvider
        {
            get
            {
                return this.modulation;
            }
            set
            {
                if (value == null)
                    throw new ArgumentNullException("Demodulation provider nemuze byt null.");
                else
                    this.modulation = value;
            }
        }

        /// <summary>
        /// Pouzity Filter Provider.
        /// </summary>
        /// <exception cref="ArgumentNullException">Pokud bylo predano null.</exception>
        public ProviderInfo FilterProvider
        {
            get
            {
                return this.filter;
            }
            set
            {
                if (value == null)
                    throw new ArgumentNullException("Filter provider nemuze byt null.");
                else
                    this.filter = value;
            }
        }

        /// <summary>
        /// Resource Descriptor pro VISA.
        /// Platny pouze pokud je dataProvider nastaven na DP_ANALYZER.
        /// </summary>
        /// <exception cref="ArgumentException">Pokud byl predan prazdny retezec.</exception>
        public string ResourceDescriptor
        {
            get
            {
                return this.resourceDescriptor;
            }

            set
            {
                if (value == "")
                {
                    throw new ArgumentException("Resource Descriptor nemůže být prázdný řetězec.");
                }
                else
                    this.resourceDescriptor = value;
            }
        }

        /// <summary>
        /// Pocet vzorku, ktere se nachytaji z analyzatoru. 
        /// Platne pouze pokud je dataProvider nastaven na DP_ANALYZER.
        /// </summary>
        /// <exception cref="ArgumentException">Pokud byl predana hodnota mensi nebo rovna nule.</exception>
        public int PocetVzorku
        {
            get
            {
                return this.pocetVzorku;
            }
            set
            {
                if (value <= 0)
                    throw new ArgumentException("Počet vzorků musí být kladné číslo.");
                else
                    this.pocetVzorku = value;
            }
        }
    }

}

