﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using AnalyzeThisForms.Components;
using Loaders.IQLoader.Embeded;
using Loaders.IQLoader.GUI;
using Loaders.IQLoader.ProviderBases;
using Loaders.IQLoader.ProviderInterfaces;
using ObjectModel.InformationElements;
using ObjectModel.Loaders;
using ObjectModel.Protocols;

///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Vstupni zasuvny modul poskytujici vstup dat ze spektralniho analyzatoru.
///

namespace Loaders.IQLoader
{
    /// <summary>
    /// Hlavni rozhrani zasuvneho modulu.
    /// </summary>
    public class IQLoaderPlugin : LoaderBase, ILoaderHasSetupDialog
    {
        /// <summary>
        /// Nastaveni pluginu, ktere je sdileno mezi nastavovacim dialogem a providery.
        /// </summary>
        private PluginSettings pluginSettings;

        /// <summary>
        /// Reference na pouzity Data provider.
        /// </summary>
        private DataProviderBase dataProvider = null;

        /// <summary>
        /// Reference na pouzity Filter provider.
        /// </summary>
        private FilterProviderBase filterProvider = null;

        /// <summary>
        /// Reference na pouzity Demodulation provider.
        /// </summary>
        private DemodulationProviderBase demodProvider = null;

        /// <summary>
        /// Reference na pouzity Protocol provider.
        /// </summary>
        private ProtocolProviderBase protProvider = null;

        /// <summary>
        /// Uchovava informace o protocol providerech nactenych z DLL.
        /// </summary>
        private Dictionary<string, ProviderInfo> protocolProviderTypes;

        /// <summary>
        /// Uchovava informace o demodulation providerech nactenych z DLL.
        /// </summary>
        private Dictionary<string, ProviderInfo> demodProviderTypes;

        /// <summary>
        /// Uchovava informace o filter providerech nactenych z DLL.
        /// </summary>
        private Dictionary<string, ProviderInfo> filterProviderTypes;

        /// <summary>
        /// Konstruktor. Provadi nacteni informaci o dostupnych providerech.
        /// </summary>
        public IQLoaderPlugin()
        {
            this.pluginSettings = new PluginSettings();
            this.protocolProviderTypes = new Dictionary<string, ProviderInfo>();
            this.demodProviderTypes = new Dictionary<string, ProviderInfo>();
            this.filterProviderTypes = new Dictionary<string, ProviderInfo>();
            this.LoadProviderClasses();
        }

        /// <summary>
        /// Pravdepodobne nevyuzivana cast rozhrani.
        /// </summary>
        /// <param name="fileName"></param>
        public IQLoaderPlugin(string fileName)
        {
        }

        /// <summary>
        /// Jmeno zasuvneho modulu.
        /// </summary>
        public override string Name
        {
            get
            {
                return "IQ Loader";
            }
        }

        /// <summary>
        /// Vytvori nastavovaci dialog a pocka dokud jej uzivatel nezavre.
        /// </summary>
        public void SetupDialog()
        {
            new Settings(this.pluginSettings, this.protocolProviderTypes, this.demodProviderTypes, this.filterProviderTypes).ShowDialog();
        }

        /// <summary>
        /// Provede overeni nastaveni pluginu. Pred OpenDevice totiz nemusel byt zavolan dialog, 
        /// ve kterem se nastaveni ulozi (uzivatel jej nemusel ani zobrazit).
        /// </summary>
        /// <returns>True pokud je vse v poradku, false pokud ne.</returns>
        private bool OverNastaveni()
        {
            // Nejprve overime, ze byli nastaveni vubec nejaci provideri.
            if (this.pluginSettings.FilterProvider == null
                || this.pluginSettings.DemodulationProvider == null
                || this.pluginSettings.ProtocolProvider == null)
            {
                new ErrorViewer("Některý provider nebyl nastaven.").ShowDialog();
                return false;
            }

            // Overime, ze vybrany demodulator typove sedi s pozadovanym.
            if (!this.pluginSettings.ProtocolProvider.ModulationType.Equals(this.pluginSettings.DemodulationProvider.ModulationType))
            {
                new ErrorViewer("Nastavená demodulace ("
                    + this.pluginSettings.DemodulationProvider.ModulationType.ToString()
                    + ") se typově neshoduje s protokolem požadovanou. ("
                    + this.pluginSettings.ProtocolProvider.ModulationType.ToString() + ").").ShowDialog();
                return false;
            }

            /*
             * Overime vzorkovaci frekvenci. Nemuze byt zaporna, protoze PluginSettings ji neumozni nastavit.
             * Nulova muze byt pouze po vytvoreni objektu, kdy je takto inicializovana. Dialog s nastavenim 
             * modulu ji vsak nastavi na nejake cislo vetsi nez nula.
             */
            if (this.pluginSettings.SampleRate == 0)
            {
                new ErrorViewer("Vzorkovací frekvence nemůže být nulová.").ShowDialog();
                return false;
            }
            return true;
        }

        /// <summary>
        /// Otevre zarizeni a inicializuje jej. 
        /// </summary>
        /// <returns>Vraci True pri uspechu, false pri chybe.</returns>
        public override bool OpenDevice()
        {
            bool statusOk = false;
            string msg = "";
            string provider = "";

            /* 
             * Nejprve overime nektera nastaveni. Je nutne to krome nastavovaciho dialogu udelat i zde,
             * protoze uzivatel nemusel nastavovaci dialog vubec vyvolat.
             */
            if (!this.OverNastaveni())
                return false;

            /*
             * Pokusime se natahnout jednotlive providery podle toho co je ulozeno v PluginSettings 
             * v providerInfo objektech.  
             */
            try
            {
                // Vytvorime data provider.
                provider = "Data provider";
                // Je to File Data Provider.
                if (this.pluginSettings.dataProvider == EDataProvider.DP_FILE)
                    this.dataProvider = new FileDataProvider(this.pluginSettings);
                else if (this.pluginSettings.dataProvider == EDataProvider.DP_ANALYZER)
                    // Je to Analyzer Data provider.
                    this.dataProvider = new AnalyzerDataProvider(this.pluginSettings);
                else
                    // Toto by nemelo nastat, pokud neni chyba v kodu.
                    throw new ProviderInitializationException("Nebyl vybrán žádný zdroj dat.");

                /*
                 * Vytvorime Filter provider. Ten je jiz ulozen v PluginSetting v podobe informace o datovem typu.
                 * Plati to i pro vestaveny filtr.
                 */
                provider = "Filter provider";
                this.filterProvider = (FilterProviderBase)Activator.CreateInstance(this.pluginSettings.FilterProvider.Typ,
                    this.pluginSettings,
                    (IQDataProvider)this.dataProvider);

                // Vytvorime Demodulation provider.
                provider = "Demod provider";
                this.demodProvider = (DemodulationProviderBase)Activator.CreateInstance(this.pluginSettings.DemodulationProvider.Typ,
                    this.pluginSettings,
                    (IQDataProvider)this.filterProvider);

                // Vytvorime Protocol provider.
                provider = "Protocol provider";
                this.protProvider = (ProtocolProviderBase)Activator.CreateInstance(this.pluginSettings.ProtocolProvider.Typ,
                    this.pluginSettings,
                    (IDemodulationProvider)this.demodProvider);

                // Pokud jsme se dostali az sem, melo by byt vsechno v poradku.
                statusOk = true;
            }
            catch (NotImplementedException ex)
            {
                msg = "Tato funkcionalita ještě není implementována:\n" + ex.Message;
            }
            catch (MemberAccessException ex)
            {
                // Pokud je napr konstruktor nastaveny jako private apod.
                msg = "Chyba při nahrávání některého provideru (" + provider + "). \n" + ex.Message;
            }
            catch (ProviderInitializationException ex)
            {
                // Doslo k chybe pri inicializacu provideru. Tato vetev nastane pravdepodobne jen u vestavenych provideru.
                msg = "Chyba při inicializaci (" + provider + "):\n" + ex.Message;
            }
            catch (TargetInvocationException ex)
            {
                /*
                 * Doslo k chybe pri inicializaci provideru. Tato vetev nastane v pripade vyjimky 
                 * u nacitanech provideru. Jedna se zde zejmena o vyjimku ProviderInitializationException.
                 */
                if (ex.InnerException is ProviderInitializationException)
                    msg = "Chyba při inicializaci (" + provider + "):\n" + ex.InnerException.Message;
                else
                    msg = ex.Message;
            }
            catch (Exception ex)
            {
                // Pro jistotu zachytavame i obecnou vyjimku.
                msg = ex.Message;
            }
            finally
            {
                // Pokud to nedopadlo dobre, musime uvolnit zdroje.
                if (!statusOk)
                {
                    new ErrorViewer(msg).ShowDialog();
                    this.DisposeAll();
                }
            }
            return statusOk;
        }

        /// <summary>
        /// Vytvori a vraci protocol factory pro nastaveny protokol.
        /// </summary>
        /// <param name="_info">Objekt nesouci typy definovanych protokolu.</param>
        /// <returns>Vraci protocol factory pro zvoleny protocol provider.</returns>
        public override ProtocolFactoryBase getProtocolFactory(ProgramInfo _info)
        {
            return this.protProvider.getProtocolFactory(_info);
        }

        /// <summary>
        /// Zrusi vsechny natahnute providery a uvolni veskere zdroje.
        /// </summary>
        private void DisposeAll()
        {
            if (this.dataProvider != null)
                this.dataProvider.CloseDevice();
            this.dataProvider = null;
            this.filterProvider = null;
            this.demodProvider = null;
            this.protProvider = null;
        }

        /// <summary>
        /// Uzavre zarizeni a uvolni vsechny prostredky.
        /// </summary>
        public override void CloseDevice()
        {
            this.DisposeAll();
        }

        /// <summary>
        /// Nacte dalsi paket ze vstupniho souboru.
        /// </summary>
        /// <returns>Dalsi paket ze vstupniho streamu nebo null pri dosazeni konce.</returns>
        public override InformationElementBase GetNextPacket()
        {
            InformationElementBase elmBase = null;
            try
            {
                byte[] dataFrame = this.protProvider.getDataFrame();
                elmBase = new InformationElementBase();
                elmBase.DataColl = new DataCollection(dataFrame);
            }
            catch (EndOfStreamException)
            {
                // Dosazen konec souboru.
                elmBase = null;
            }
            catch (NullReferenceException)
            {
                // Toto by za normalnich okolnosti nemelo nastat.
                new ErrorViewer("Interní chyba").ShowDialog();
            }
            catch (FileFormatException ex)
            {
                // Pri nacitani ze souboru se narazilo na neco, co neodpovida formatu. Taky musime skoncit.
                new ErrorViewer(ex.Message).ShowDialog();
            }
            catch (Exception ex)
            {
                // Jina chyba.
                new ErrorViewer(ex.Message).ShowDialog();
            }
            return elmBase;
        }

        /// <summary>
        /// Nalezne a nacte informace o providerech z adresare "plugins".
        /// </summary>
        private void LoadProviderClasses()
        {
            this.protocolProviderTypes.Clear();
            this.demodProviderTypes.Clear();
            this.filterProviderTypes.Clear();

            // Pridame vychozi filtr, ktery je zabudovany primo do pluginu.
            Type pomTyp = typeof(DefaultFilter);
            this.filterProviderTypes.Add(pomTyp.ToString(), new ProviderInfo(DefaultFilter.getFilterName(), pomTyp));

            string providerDir = "plugins";

            // zpravujeme kazdy soubor v adresari providerDir
            foreach (string file in System.IO.Directory.GetFiles(providerDir, "*.dll"))
            {
                string fullpath = System.IO.Path.Combine(Directory.GetCurrentDirectory(), file);
                Assembly assembly = Assembly.LoadFile(fullpath);

                // pro vsechny typy v assembly
                foreach (Type type in assembly.GetTypes())
                {
                    // Neverejne typy preskakujeme.
                    if (!type.IsClass || type.IsNotPublic) continue;

                    // Protocol provider.
                    if (type.IsSubclassOf(typeof(ProtocolProviderBase)))
                    {
                        // Pokud jsme provider s takovym nazvem jiz nasli,preskocime.
                        if (this.protocolProviderTypes.ContainsKey(type.FullName))
                            continue;

                        string name = (string)type.GetMethod("getProtocolName").Invoke(this, null);
                        EModulationType modulation = (EModulationType)type.GetMethod("getModulationType").Invoke(this, null);
                        float bw = (float)type.GetMethod("getBandwidth").Invoke(this, null);

                        // Ulozime si ziskane informace.
                        this.protocolProviderTypes.Add(type.FullName,
                            new ProviderInfo(name, modulation, type, bw));
                    }
                    // Demodulation provider.
                    else if (type.IsSubclassOf(typeof(DemodulationProviderBase)))
                    {
                        // Pokud jsme provider s takovym nazvem jiz nasli, preskocime.
                        if (this.demodProviderTypes.ContainsKey(type.FullName))
                            continue;

                        string name = (string)type.GetMethod("getDemodName").Invoke(this, null);
                        EModulationType modulation = (EModulationType)type.GetMethod("getModulationType").Invoke(this, null);

                        // Ulozime si ziskane informace.
                        this.demodProviderTypes.Add(type.FullName,
                            new ProviderInfo(name, modulation, type));
                    }
                    // Filter provider.
                    else if (type.IsSubclassOf(typeof(FilterProviderBase)))
                    {
                        // Pokud jsme provider s takovym nazvem jiz nasli,preskocime.
                        if (this.filterProviderTypes.ContainsKey(type.FullName))
                            continue;

                        string name = (string)type.GetMethod("getFilterName").Invoke(this, null);

                        // Ulozime si ziskane informace.
                        this.filterProviderTypes.Add(type.FullName,
                            new ProviderInfo(name, type));
                    }
                }
            }
        }
    }
}
