﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Windows.Forms;

///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Vstupni zasuvny modul poskytujici vstup dat ze spektralniho analyzatoru.
///

namespace Loaders.IQLoader.GUI
{
    public partial class SettingsPanel : UserControl
    {
        /// <summary>
        /// Instance nastaveni zasuvneho modulu.
        /// </summary>
        private PluginSettings settings;

        /// <summary>
        /// Informace o nactenych Protocol providerech
        /// </summary>
        private Dictionary<string, ProviderInfo> protocolProviderTypes;

        /// <summary>
        /// Informace o nactech Demodulation providerech.
        /// </summary>
        private Dictionary<string, ProviderInfo> demodProviderTypes;

        /// <summary>
        /// Informace o nactenych Filter providerech.
        /// </summary>
        private Dictionary<string, ProviderInfo> filterProviderTypes;

        /// <summary>
        /// Konstruktor.
        /// </summary>
        public SettingsPanel()
        {
            InitializeComponent();
        }

        /// <summary>
        /// Je nutne aby byla zavolana co nejdrive po vytvoreni objektu. Nastavi ruzne interni stuktury.
        /// </summary>
        /// <param name="settings">Objekt uchovavajici nastaveni pluginu.</param>
        /// <param name="protocolProviderTypes">Seznam nactenych protocol provideru</param>
        /// <param name="demodProviderTypes">Seznam nactenych demod provideru.</param>
        /// <param name="filterProviderTypes">Seznam nactenych filter provideru.</param>
        public void SetStructures(PluginSettings settings,
            Dictionary<string, ProviderInfo> protocolProviderTypes,
            Dictionary<string, ProviderInfo> demodProviderTypes,
            Dictionary<string, ProviderInfo> filterProviderTypes)
        {
            this.settings = settings;
            this.protocolProviderTypes = protocolProviderTypes;
            this.demodProviderTypes = demodProviderTypes;
            this.filterProviderTypes = filterProviderTypes;

            // set up components
            UpdateDataProviderRadioButtons();
            UpdateProtocolComboBox();
            UpdateDemodComboBox();
            UpdateFilterComboBox();

            this.txtVzorkovaciFrekvence.Text = this.settings.SampleRate.ToString();
            this.txtNosnaFrekvence.Text = this.settings.CenterFrequency.ToString();
            this.txtPocetVzorku.Text = this.settings.PocetVzorku.ToString();
            this.txtRefLevel.Text = this.settings.RefLevel.ToString();
            this.txtResourceDescriptor.Text = this.settings.ResourceDescriptor;
            this.txtSirkaPasma.Text = this.settings.Bandwith.ToString();
            this.txtSoubor.Text = this.settings.FilePath;
        }

        /// <summary>
        /// Nastavuje radio buttony podle toho co je v settings. Vola se pri volani SetStructures().
        /// </summary>
        /// <exception cref="InvalidOperationException">Pokud je nastaven neznamy zdroj dat(DP_NONE).</exception>
        private void UpdateDataProviderRadioButtons()
        {
            if (this.settings.dataProvider == EDataProvider.DP_FILE)
                this.rdbZeSouboru.Checked = true;
            else if (this.settings.dataProvider == EDataProvider.DP_ANALYZER)
                this.rdbAnalyzator.Checked = true;
            else
                this.rdbZeSouboru.Checked = true;
        }

        /// <summary>
        /// Updates item in Protocol combo box
        /// </summary>
        private void UpdateProtocolComboBox()
        {
            if (this.protocolProviderTypes.Count == 0)
            {
                this.cmbProtokol.Enabled = false;
            }
            else
            {
                this.cmbProtokol.Enabled = true;
                this.cmbProtokol.Items.Clear();

                foreach (var item in this.protocolProviderTypes)
                {
                    KeyValuePair<string, ProviderInfo> pom = item;
                    this.cmbProtokol.Items.Add(pom.Value);
                }
                this.cmbProtokol.DisplayMember = "Name";
                this.cmbProtokol.ValueMember = "Typ";

                if (this.settings.ProtocolProvider == null)
                {
                    this.cmbProtokol.SelectedIndex = 0;
                }
                else
                {
                    this.cmbProtokol.SelectedIndex = this.cmbProtokol.FindStringExact(this.settings.ProtocolProvider.Name);
                }
            }
        }

        /// <summary>
        /// Aktualizuje polozky v combo boxu s demodulatory.
        /// </summary>
        private void UpdateDemodComboBox()
        {
            if (this.demodProviderTypes.Count == 0)
            {
                this.cmbDemod.Enabled = false;
            }
            else
            {
                this.cmbDemod.Enabled = true;
                this.cmbDemod.Items.Clear();

                foreach (var item in this.demodProviderTypes)
                {
                    KeyValuePair<string, ProviderInfo> pom = item;
                    this.cmbDemod.Items.Add(pom.Value);
                }
                this.cmbDemod.DisplayMember = "Name";
                this.cmbDemod.ValueMember = "Typ";

                if (this.settings.DemodulationProvider == null)
                {
                    this.cmbDemod.SelectedIndex = 0;
                }
                else
                {
                    this.cmbDemod.SelectedIndex = this.cmbDemod.FindStringExact(this.settings.DemodulationProvider.Name);
                }
            }
        }

        /// <summary>
        /// Aktualizuje polozky v combo boxu pro filtry.
        /// </summary>
        private void UpdateFilterComboBox()
        {
            if (this.filterProviderTypes.Count == 0)
            {
                this.cmbFiltr.Enabled = false;
            }
            else
            {
                this.cmbFiltr.Enabled = true;
                this.cmbFiltr.Items.Clear();

                foreach (var item in this.filterProviderTypes)
                {
                    KeyValuePair<string, ProviderInfo> pom = item;
                    this.cmbFiltr.Items.Add(pom.Value);
                }

                this.cmbFiltr.DisplayMember = "Name";
                this.cmbFiltr.ValueMember = "Typ";

                if (this.settings.FilterProvider == null)
                {
                    this.cmbFiltr.SelectedIndex = 0;
                }
                else
                {
                    this.cmbFiltr.SelectedIndex = this.cmbFiltr.FindStringExact(this.settings.FilterProvider.Name);
                }
            }
        }

        /// <summary>
        /// Nastavi vlastnosti enabled u ovladacich prvku tak, aby to odpovidalo volbe zdroje dat ze souboru.
        /// </summary>
        private void selectSourceFromFile()
        {
            this.txtResourceDescriptor.Enabled = false;
            this.lblResourceDescriptor.Enabled = false;
            this.txtPocetVzorku.Enabled = false;
            this.lblPocetVzorku.Enabled = false;
            this.txtNosnaFrekvence.Enabled = false;
            this.lblNosnaFrekvence.Enabled = false;
            this.txtSirkaPasma.Enabled = false;
            this.lblMerenaSirkaPasma.Enabled = false;
            this.txtRefLevel.Enabled = false;
            this.lblReferenceLevel.Enabled = false;
            this.txtSoubor.Enabled = true;
            this.lblSoubor.Enabled = true;
            this.btnProchazet.Enabled = true;
        }

        /// <summary>
        /// Nastavi vlastnosti enabled u ovladacich prvku tak, aby to odpovidalo volbe zdroje dat z analyzatoru.
        /// </summary>
        private void selectSourceFromAnalyzer()
        {
            this.txtResourceDescriptor.Enabled = true;
            this.lblResourceDescriptor.Enabled = true;
            this.txtPocetVzorku.Enabled = true;
            this.lblPocetVzorku.Enabled = true;
            this.txtNosnaFrekvence.Enabled = true;
            this.lblNosnaFrekvence.Enabled = true;
            this.txtSirkaPasma.Enabled = true;
            this.lblMerenaSirkaPasma.Enabled = true;
            this.txtRefLevel.Enabled = true;
            this.lblReferenceLevel.Enabled = true;
            this.txtSoubor.Enabled = false;
            this.lblSoubor.Enabled = false;
            this.btnProchazet.Enabled = false;
        }

        private void btnProchazet_Click(object sender, EventArgs e)
        {
            if (this.openFileDialog.ShowDialog() == DialogResult.OK)
            {
                this.txtSoubor.Text = this.openFileDialog.FileName;
            }

        }

        private void cmbProtokol_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (this.cmbProtokol.SelectedIndex != -1)
            {
                ProviderInfo provider = (ProviderInfo)this.cmbProtokol.SelectedItem;
                this.lblTypModulace.Text = provider.ModulationType.ToString();
                this.lblSirkaPasma.Text = provider.Bw.ToString() + " Hz";
            }
        }

        private void rdbZeSouboru_CheckedChanged(object sender, EventArgs e)
        {
            this.selectSourceFromFile();
        }

        private void rdbAnalyzator_CheckedChanged(object sender, EventArgs e)
        {
            this.selectSourceFromAnalyzer();
        }

        /// <summary>
        /// Provede overeni zadaneho nastaveni a jeho ulozeni do PluginSettings 
        /// </summary>
        /// <returns>True pokud probehlo v pordku, false pokud ne.</returns>
        public bool VerifyAndSave()
        {
            string msg = "";
            bool statusOK = true;
            int sampleRate = 0;
            int pocetVzorku = 0;
            int nosna = 0;
            int sirkaPasma = 0;
            double refLevel = 0;

            // overime, ze je vybrana nejaka demodulace
            if (this.cmbDemod.SelectedIndex == -1)
            {
                msg += "Nebyla vybrána žádná demodulace.\n";
                statusOK = false;
            }

            // overime, ze je vybran nejaky protokol
            if (this.cmbProtokol.SelectedIndex == -1)
            {
                msg += "Nebyla vybrán žádný Protocol Provider.\n";
                statusOK = false;
            }

            // filtr by mel byt vybran vzdy (protoze existuje defaultni), ale presto overime
            // overime, ze je vybran nejaky protokol
            if (this.cmbFiltr.SelectedIndex == -1)
            {
                msg += "Nebyl vybrán žádný Filter Provider.\n";
                statusOK = false;
            }

            /*
             * Nyni overime, ze vybrana demodulace se shoduje s tou, kterou pozaduje protokol. 
             * Overovani na null je tam kvuli tomu, ze protocol a demod se sice overili v predchozim kroku, 
             * ale tento krok se provede nezavisle na tom, jak ty ostatni dopadly.
            */
            ProviderInfo protocol = (ProviderInfo)this.cmbProtokol.SelectedItem;
            ProviderInfo demod = (ProviderInfo)this.cmbDemod.SelectedItem;
            if (protocol != null && demod != null && !protocol.ModulationType.Equals(demod.ModulationType))
            {
                msg += "Protocol provider požaduje jiný typ demodulace (" + protocol.ModulationType.ToString() + ").\n";
                statusOK = false;
            }

            // Prevedeme vzorkovaci frekvenci a overime ji.
            try
            {
                sampleRate = Convert.ToInt32(this.txtVzorkovaciFrekvence.Text);
                if (sampleRate <= 0)
                {
                    msg += "Vzorkovací frekvence musí být kladné číslo.\n";
                    statusOK = false;
                }
            }
            catch (FormatException)
            {
                msg += "Vzorkovací frekvence musí být celé kladné číslo.\n";
                statusOK = false;
            }


            // Nyni provedeme overeni na zaklade vybraneho zdroje dat.
            if (this.rdbZeSouboru.Checked)
            {
                // Nazev souboru.
                if (this.txtSoubor.Text == "")
                {
                    msg += "Nebyla zadána cesta k souboru.\n";
                    statusOK = false;
                }
            }
            else if (this.rdbAnalyzator.Checked)
            {
                // Resource descriptor
                if (this.txtResourceDescriptor.Text == "")
                {
                    msg += "Nebyl zadán Resource Descriptor.\n";
                    statusOK = false;
                }

                // Pocet vzorku
                try
                {
                    pocetVzorku = Convert.ToInt32(this.txtPocetVzorku.Text);
                    if (pocetVzorku <= 0)
                    {
                        msg += "Počet vzorků musí být kladné číslo.\n";
                        statusOK = false;
                    }
                }
                catch (FormatException)
                {
                    msg += "Pole Počet vzorků nebylo vyplněno nebo obsahuje chybu.\n";
                    statusOK = false;
                }

                // Nosna frekvence
                try
                {
                    nosna = Convert.ToInt32(this.txtNosnaFrekvence.Text);
                    if (nosna <= 0)
                    {
                        msg += "Nosná frekvence musí být kladné číslo.\n";
                        statusOK = false;
                    }

                }
                catch (FormatException)
                {
                    msg += "Pole nosná frekvence nebylo vyplněno nebo obsahuje chybu.\n";
                    statusOK = false;
                }

                // Sirka pasma
                try
                {
                    sirkaPasma = Convert.ToInt32(this.txtSirkaPasma.Text);
                    if (nosna <= 0)
                    {
                        msg += "Šířka pásma musí být kladné číslo.\n";
                        statusOK = false;
                    }
                }
                catch (FormatException)
                {
                    msg += "Pole šířka pásma nebylo vyplněno nebo obsahuje chybu.\n";
                    statusOK = false;
                }

                // Reference level
                try
                {
                    NumberFormatInfo nf = new CultureInfo("en-US", false).NumberFormat;
                    refLevel = Convert.ToDouble(this.txtRefLevel.Text, nf);
                }
                catch (FormatException)
                {
                    msg += "Pole Reference level nebylo vyplněno nebo obsahuje chybu.\n";
                    statusOK = false;
                }
            }
            else
            {
                msg += "Interní chyba. Zdroj dat nenastaven.\n";
                statusOK = false;
            }

            // Pokud vse probehlo spravne, ulozime nastaveni.
            if (statusOK)
            {
                this.settings.ProtocolProvider = protocol;
                this.settings.DemodulationProvider = demod;
                this.settings.FilterProvider = (ProviderInfo)this.cmbFiltr.SelectedItem;
                this.settings.SampleRate = sampleRate;

                if (this.rdbZeSouboru.Checked)
                {
                    this.settings.dataProvider = EDataProvider.DP_FILE;
                    this.settings.FilePath = this.txtSoubor.Text;
                }
                else
                {
                    this.settings.dataProvider = EDataProvider.DP_ANALYZER;
                    this.settings.CenterFrequency = nosna;
                    this.settings.ResourceDescriptor = txtResourceDescriptor.Text;
                    this.settings.PocetVzorku = pocetVzorku;
                    this.settings.Bandwith = sirkaPasma;
                    this.settings.RefLevel = refLevel;
                }
            }
            else
            {
                // Nekde doslo k chybe(am).
                new ErrorViewer(msg).ShowDialog();
            }
            return statusOK;
        }
    }
}
