﻿using System;
using System.IO;
using InstrumentDrivers;
using Loaders.IQLoader.ProviderBases;

///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Vstupni zasuvny modul poskytujici vstup dat ze spektralniho analyzatoru.
///

namespace Loaders.IQLoader.Embeded
{
    /// <summary>
    /// Cte I/Q body pomoci primo z analyzatoru pres rozhrani VISA pomoci 
    /// ovladace RSSpecAn od firmy Rohde&Swarz.
    /// </summary>
    public class AnalyzerDataProvider : DataProviderBase
    {
        /// <summary>
        /// Timeout pro zachytavani dat v ms.
        /// </summary>
        private const int timeOut = 20000;

        /// <summary>
        /// Pole pro ulozeni fazove slozky signalu.
        /// </summary>
        private double[] resultI = null;

        /// <summary>
        /// Pole pro ulozeni kvadraturni slozky signalu.
        /// </summary>
        private double[] resultQ = null;

        /// <summary>
        /// Handle analyzatoru.
        /// </summary>
        private rsspecan analyzator = null;

        /// <summary>
        /// Obsahuje pocet skutecne namerenych hodnot.
        /// </summary>
        private int pocetVracenychHodnot = 0;

        /// <summary>
        /// Ukazuje na dalsi polozku v poli dat, ktera bude vracena volanim getIQPoint().
        /// </summary>
        private int index = 0;

        /// <summary>
        /// Konstruktor. Inicializuje ovladac a provede nastaveni spektralniho analyzatoru.
        /// Dale nacte data do cache.
        /// </summary>
        /// <param name="settings">Instance Plugin settings s nastavenim zasuvneho modulu.</param>
        /// <exception cref="ProviderInitializationException">Pokud doslo pri inicializaci provideru k chybe.</exception>
        public AnalyzerDataProvider(PluginSettings settings)
            : base(settings)
        {
            try
            {
                // Pole pro ulozeni I a Q bude mit velikost podle toho, kolik pozadujeme vzorku.
                this.resultI = new double[settings.PocetVzorku];
                this.resultQ = new double[settings.PocetVzorku];

                // Inicializujeme ovladac.
                this.analyzator = new rsspecan(settings.ResourceDescriptor, true, true);

                // Nastavime nosnou.
                this.analyzator.ConfigureFrequencyCenter(1, Convert.ToDouble(settings.CenterFrequency));

                // Nastavime reference level.
                this.analyzator.ConfigureReferenceLevel(1, settings.RefLevel);

                // Nastavime, ze budeme chytat IQ.
                this.analyzator.ConfigureTraceIQDataAcquisition(1, true);

                // Nastavime vzorkovaci frekvenci a ostatni parametry.
                this.analyzator.TraceIQSet(1, Convert.ToDouble(settings.SampleRate), 0, 0, Convert.ToDouble(settings.Bandwith), 0, settings.PocetVzorku);

                // Provedeme vlastni zachyceni dat.
                this.analyzator.ReadTraceIQData(1, timeOut, settings.PocetVzorku, out this.pocetVracenychHodnot, this.resultI, this.resultQ);

            }
            catch (System.Runtime.InteropServices.ExternalException ex)
            {
                throw new ProviderInitializationException("Chyba při praci ovladace. VISA hlásí:\n" + ex.Message + "\n\nChybový kód: " + ex.ErrorCode.ToString());
            }
        }

        #region IQDataProvider Members
        /// <summary>
        /// Vraci dalsi nacteny I/Q bod.
        /// </summary>
        /// <returns>Vraci dalsi I/Q bod.</returns>
        /// <exception cref="EndOfStreamException">Jakmile je dosazeno konce dat.</exception>
        public override IQData getIQPoint()
        {
            if (this.index >= this.pocetVracenychHodnot)
                throw new EndOfStreamException();

            IQData tmp = new IQData(this.resultI[this.index], this.resultQ[this.index]);
            this.index++;
            return tmp;
        }

        /// <summary>
        /// Uvolni prostredky.
        /// </summary>
        public override void CloseDevice()
        {
            this.resultI = null;
            this.resultQ = null;
            this.pocetVracenychHodnot = 0;
            this.index = 0;
            this.analyzator = null;
        }

        #endregion
    }
}
