﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using AnalyzeThisForms.Components;
using Loaders.IQLoader.ProviderBases;
using Loaders.IQLoader.ProviderInterfaces;
using ObjectModel.Protocols;

///
/// (c)2010 Tomas Ocelik
/// xoceli00@fit.vutbr.cz
/// 
/// Protocol provider pro sit MPT-1327.
///

namespace Loaders.IQLoader.mpt1327
{
    /// <summary>
    /// Protocol provider pro MPT-1327.
    /// </summary>
    public class mpt1327provider : ProtocolProviderBase
    {
        /// <summary>
        /// Delka kodoveho slova v bajtech.
        /// </summary>
        private const int frameLen = 8;

        /// <summary>
        /// Delka synchronizacniho useku v bitech.
        /// </summary>
        private const int synchBitsLen = 16;

        /// <summary>
        /// Delka codeword synchronization sequence v bitech.
        /// </summary>
        private const int cdwSynchLen = 16;

        /// <summary>
        /// Bitova rychlost v bitech za sekundu.
        /// </summary>
        private const int bitRate = 1200;

        /// <summary>
        /// Prah pro bitovou synchronizaci v poctech vzorku.
        /// </summary>
        private int synchTreshold = 0;

        /// <summary>
        /// Prah pro codeword synchronization sequence v poctech vzorku.
        /// </summary>
        private int cdwSyncTreshold = 0;

        /// <summary>
        /// Pocet vzorku na symbol.
        /// </summary>
        private int pocetVzorkuBit = 0;

        /// <summary>
        /// Delka bufferu pro hledani synchronizace. Pocita se jako synchBitsLen * pocetVzorkuSymb.
        /// </summary>
        private int bufLen = 0;

        /// <summary>
        /// Urcuje, zda uz byla nalezena bitova synchronizace - hleda se pouze pri prvnim spusteni.
        /// </summary>
        private bool bitSyncFound = false;

        /// <summary>
        /// Konstruktor.
        /// </summary>
        /// <param name="settings">Instance of plugin settings.</param>
        /// <param name="demodProvider">Instance of Demodulation provider.</param>
        /// <exception cref="ProviderInitializationException">Pokud dojde pri inicializaci k chybe.</exception>
        /// <exception cref="ArgumentNullException">Pokud je nektery z parametru null.</exception>
        public mpt1327provider(PluginSettings settings, IDemodulationProvider demodProvider)
            : base(settings, demodProvider)
        {
            /*
             * Podle vzorkovaci frekvence a bitove rychlosti si spocitame kolik vzorku vychazi
             * na jeden bit. Mel by to byt lichy pocet a cele cislo.
             */
            this.pocetVzorkuBit = settings.SampleRate / bitRate;
            if (settings.SampleRate % bitRate != 0)
                throw new ProviderInitializationException("Vzorkovaci frekvence musi byt nasobkem bitove rychlosti");
            if (this.pocetVzorkuBit % 2 == 0)
                throw new ProviderInitializationException("Vzorkovaci frekvence musi byt zvolena tak, aby vychazel lichy pocet vzorku na bit.");

            if (this.pocetVzorkuBit == 1)
                MessageBox.Show("Varování: Vychází jeden vzorek na bit, což může vést k velké chybovosti.");

            // nyni spocitame prahy pro bitovou synchronizace a SYNC slovo
            // pro bitovou synchronizaci to bude 10%
            this.synchTreshold = (int)Math.Round(this.pocetVzorkuBit * synchBitsLen * 0.1);

            // pro SYNC codeword to bude taky 10%
            this.cdwSyncTreshold = (int)Math.Round(cdwSynchLen * 0.1);

            // Spocitame velikost vstupnuho bufferu. Bitomva synchronizace ma 16bitu * pocetVzorkuNa Symbol
            this.bufLen = mpt1327provider.synchBitsLen * this.pocetVzorkuBit;
        }

        /// <summary>
        /// Vraci jmeno protokolu.
        /// </summary>
        /// <returns>Vraci jmeno protokolu.</returns>
        public static new string getProtocolName()
        {
            return "MPT-1327";
        }

        /// <summary>
        /// Vraci typ modulace pozadovany protokolem.
        /// </summary>
        /// <returns>Vraci typ modulace pozadovany protokolem.</returns>
        public static new EModulationType getModulationType()
        {
            return EModulationType.MT_FSK;
        }

        /// <summary>
        /// Vraci sirku pasma.
        /// </summary>
        /// <returns>Vraci sirku pasma.</returns>
        public static new float getBandwidth()
        {
            return 25000;
        }

        /// <summary>
        /// Vraci dalsi datovy ramec ze svstupniho proudu dat.
        /// </summary>
        /// <returns>Vraci dalsi datovy ramec ze svstupniho proudu dat.</returns>
        /// <exception cref="EndOfStreamException">Pokud je dosazeno konce souboru.</exception>  
        public override byte[] getDataFrame()
        {
            // Zjistime bitovou synchronizaci a najdeme control chanel codeword sequence.
            if (!this.bitSyncFound)
            {
                do
                {
                    this.findBitSync();
                }
                while (!this.findCodewordSeq());
                this.bitSyncFound = true;
            }

            // Precteme jeden codeword. Musime bity skladat do bajtu po osmi.
            byte[] codewordBuffer = new byte[frameLen];
            for (int i = 0; i < frameLen; i++)
            {
                byte bajt = 0;
                for (int j = 0; j < 8; j++)
                {
                    bajt <<= 1;
                    bajt |= this.readNextBit();
                }
                System.Console.Out.WriteLine(Convert.ToString(bajt, 16) + "|");
                codewordBuffer[i] = bajt;
            }
            return codewordBuffer;
        }

        /// <summary>
        /// Vytvori protocol factory pro MPT-1327.
        /// </summary>
        /// <param name="info_">Objekt nesouci definovane protokoly.</param>
        /// <returns>Vraci protocol factory pro MPT-1327.</returns>
        public override ProtocolFactoryBase getProtocolFactory(ProgramInfo info_)
        {
            return new Mpt1327ProtocolFactory(info_);
        }

        /// <summary>
        /// Hleda bitovou synchronizaci. Pro MPT-1327 je bitova synchronizace odvozena z 
        /// preambule, ktera je 16 bitova a obsahuje prechody 10 v kazdem itenravalu jednickou pocinaje
        /// a nulou konce.
        /// </summary>
        private void findBitSync()
        {
            List<byte> inputBuffer = new List<byte>();
        again:
            int bufLen = inputBuffer.Count;
            // Naplnime buffer tak, aby byl presne delky preambule 
            int bufSizeMax = this.bufLen;
            while (bufLen < bufSizeMax)
            {
                inputBuffer.Add(this.demodProvider.getBit());
                bufLen++;
            }

            // Pokusime se aplikovat bitovou synchronizaci - pouze na tu cast, ktera by mela byt preambuli
            int i = 0;
            int countOfBad = 0;
            while (i < this.bufLen)
            {
                // Urcity pocet vzorku budou jednicky.
                int j;
                for (j = i; j < i + this.pocetVzorkuBit; j++)
                {
                    //System.Console.Out.Write(inputBuffer[j].ToString());
                    if (!inputBuffer[j].Equals(1))
                        countOfBad++;
                }
                i = j;

                // Urcity pocet vzorku budou nuly.
                for (j = i; j < i + this.pocetVzorkuBit; j++)
                {
                    //System.Console.Out.Write(inputBuffer[j].ToString());
                    if (!inputBuffer[j].Equals(0))
                        countOfBad++;
                }
                i = j;
            }
            System.Console.Out.WriteLine(" |> " + countOfBad.ToString());

            // Pokud nactena data nevyhouje pozadovany kriteriim, smazeme z leva jeden vzorek, obsah bufferu posuneme a nacteme dalsi vzorek
            if (countOfBad > this.synchTreshold)
            {
                inputBuffer.RemoveAt(0);
                goto again;
            }
        }

        /// <summary>
        /// Pokusi se najit codeword synchronization sequence (SYNC) 1100010011010111
        /// </summary>
        /// <returns>True pokud bylo SYNC nalezeno, False pokud ne.</returns>
        private bool findCodewordSeq()
        {
            byte[] modelCodeword = { 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 1, 1 };
            int badBitCnt;
            List<byte> inputBuffer = new List<byte>();
            int loop = 0;

            System.Console.Out.Write("|");
        again:

            badBitCnt = 0;
            int bufLen = inputBuffer.Count;
            // Naplnime buffer tak, aby byl presne delky SYNC
            while (bufLen < cdwSynchLen)
            {
                inputBuffer.Add(this.readNextBit());
                bufLen++;
            }

            for (int i = 0; i < modelCodeword.Length; i++)
            {
                System.Console.Out.Write(inputBuffer[i].ToString());
                if (!inputBuffer[i].Equals(modelCodeword[i]))
                    badBitCnt++;
            }

            System.Console.Out.WriteLine("|");
            if (badBitCnt <= cdwSyncTreshold)
                return true;

            if (loop < 96)
            {
                inputBuffer.RemoveAt(0);
                loop++;
                goto again;
            }
            else
                return false;
        }

        /// <summary>
        /// Precte dalsi bit ze synchronizovaneho vstupniho proudu dat. 
        /// Zde jiz se jedna opravdu o datove bity. Hodnota bitu je urcena podle toho, 
        /// ktery typ vzorku prevazoval.
        /// </summary>
        /// <returns>Vraci dalsi bit ze vstupniho proudu.</returns>
        private byte readNextBit()
        {
            int pocetJednicek = 0;
            int pocetNul = 0;
            byte bit;

            for (int i = 0; i < this.pocetVzorkuBit; i++)
            {
                bit = this.demodProvider.getBit();
                System.Console.Out.Write(bit.ToString());
                if (bit == 1)
                    pocetJednicek++;
                else
                    pocetNul++;
            }
            System.Console.Out.Write("|");

            // Nemuze nastat rovnost, protoze pocet vzorku / symbol musi byt liche cislo.
            if (pocetNul > pocetJednicek)
                return 0;
            else
                return 1;
        }
    }
}
